<?php

namespace Atlas\RandomisationBundle\Controller;

use Atlas\RandomisationBundle\Exception\ReportException;
use Atlas\RandomisationBundle\Repository\Randomisation\AllocationRepository;
use Atlas\RandomisationBundle\Repository\Study\RandomisationRepository;
use Atlas\RandomisationBundle\Security\RandomisationAuthorisationInterface;
use Atlas\RandomisationBundle\Service\Report\ReportBuilder;
use Atlas\ReportRenderBundle\Service\ReportRenderer;
use Psr\Cache\InvalidArgumentException;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\Component\Uid\Uuid;

#[IsGranted('ROLE_USER')]
final class ReportController extends AbstractController
{
    /**
     * @param RandomisationAuthorisationInterface $authorisation
     */
    public function __construct(
        private readonly RandomisationAuthorisationInterface $authorisation,
    )
    {
    }

    #[Route('/report/randomisation/study/{study<[A-Za-z0-9_-]+>}', name: 'shared_report_rand', methods: ['GET'])]
    public function index(
        string $study,
        RandomisationRepository $randomisation,
        AllocationRepository $allocations
    ): Response
    {
        if (!$this->authorisation->viewReport($study)) {
            throw new AccessDeniedHttpException();
        }

        if ($this->authorisation->isBlinded($study)) {
            throw new AccessDeniedHttpException();
        }

        $list = $randomisation->findByStudyCode($study);

        $simulations = $allocations->getSimulationMapForStudy($study);

        return $this->render('@Randomisation/report/index.html.twig', [
            'study' => $study,
            'randomisation' => $list,
            'simulations' => $simulations,
        ]);
    }

    /**
     * @param Request $request
     * @param string $study
     * @param string $location
     * @param string $code
     * @param ReportBuilder $reporter
     * @param ReportRenderer $renderer
     * @return Response
     * @throws InvalidArgumentException
     */
    #[Route(
        '/report/{code<[A-Za-z0-9_-]+>}/study/{study<[A-Za-z0-9_-]+>}',
        name: 'shared_rand_report_page',
        defaults: ['_uppercase' => ['code', 'specification', 'study']],
        methods: ['GET'])
    ]
    public function report(
        Request $request,
        string $study,
        string $code,
        ReportBuilder $reporter,
        ReportRenderer $renderer
    ): Response
    {

        if (!$this->authorisation->viewReport($study)) {
            throw new AccessDeniedHttpException();
        }

        if ($this->authorisation->isBlinded($study)) {
            throw new AccessDeniedHttpException();
        }

        $simulated = $request->query->get('simulated');

        if($simulated) {
            if (!Uuid::isValid($simulated)) {
                throw new ReportException('Invalid simulation id');
            } else {
                $simulated = Uuid::fromString($simulated);
            }
        }

        $layout = $reporter->generate($study, $code, simulated: $simulated);

        return $renderer->renderHtmlResponse($layout);

    }
}
