<?php

declare(strict_types=1);

namespace Atlas\RandomisationBundle\Entity\Randomisation;

use Atlas\AuditBundle\Attribute\AuditActor;
use Atlas\AuditBundle\Attribute\AuditTimestamp;
use Atlas\AuditBundle\Attribute\Enum\AuditActionType;
use Atlas\RandomisationBundle\Exception\NotBlankException;
use Atlas\RandomisationBundle\Exception\RandomisationException;
use Atlas\RandomisationBundle\Repository\Randomisation\AllocationRepository;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Types\UuidType;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Uid\Uuid;

#[ORM\Entity(repositoryClass: AllocationRepository::class)]
#[ORM\Table(
    name: 'rand_allocation',
    indexes: [
        new ORM\Index(columns: ['study_code']),
        new ORM\Index(columns: ['randomisation_code']),
        new ORM\Index(columns: ['participant_identifier']),
        new ORM\Index(columns: ['arm_allocated']),
        new ORM\Index(columns: ['run_id']),
        new ORM\Index(columns: ['simulation_id'])
    ])
]
#[ORM\UniqueConstraint(columns: ['study_code', 'randomisation_code', 'participant_identifier', 'simulation_id'])]
#[UniqueEntity(fields: ['study_code', 'randomisation_code', 'participant_identifier', 'simulation_id'])]
class Allocation
{

    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\Column]
    public string $study_code;

    #[ORM\Column]
    private(set) string $randomisation_code;

    #[ORM\Column]
    private(set) string $participant_identifier;

    #[ORM\Column(options: ['default' => 1])]
    private(set) int $version = 1;

    #[ORM\Column(nullable: true)]
    private(set) ?string $randomisation_group = null;

    #[ORM\Column(nullable: true)]
    private(set) ?string $type = null; //e.g. manual

    #[ORM\Column]
    private(set) string $arm_allocated;

    #[ORM\Column(type: UuidType::NAME)]
    private(set) Uuid $run_id;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[AuditTimestamp(AuditActionType::INSERT)]
    private(set) DateTimeInterface $randomised_on;

    #[ORM\Column(length: 255)]
    #[AuditActor(AuditActionType::INSERT)]
    private(set) string $randomised_by;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE, nullable: true, options: ['default' => null])]
    private(set) ?DateTimeInterface $simulation = null;

    #[ORM\Column(type: UuidType::NAME, nullable: true, options: ['default' => null])]
    private(set) ?Uuid $simulation_id = null;

    //not in db
    #[AuditActor(AuditActionType::DELETE)]
    private(set) string $deleted_by;
    #[AuditTimestamp(AuditActionType::DELETE)]
    private(set) DateTimeInterface $deleted;

    /**
     * @param string $studyCode
     * @param string $randomisationCode
     * @param string $participantIdentifier
     * @param string $armAllocated
     * @param int $version
     * @param Uuid $runId
     * @param string $actionBy
     * @param string|null $type for example manual or minimised
     * @param string|null $randomisationGroup
     * @param DateTimeInterface|null $timestamp
     * @param DateTimeInterface|null $simulation
     * @param Uuid|null $simulationId
     */
    public function __construct(
        string $studyCode,
        string $randomisationCode,
        string $participantIdentifier,
        string $armAllocated,
        int $version,
        Uuid $runId,
        string $actionBy,
        ?string $type = null,
        ?string $randomisationGroup = null,
        ?DateTimeInterface $timestamp = null,
        ?DateTimeInterface $simulation = null,
        ?Uuid $simulationId = null
    )
    {
        if(empty($studyCode = mb_trim($studyCode, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Study code');
        }

        if(empty($randomisationCode = mb_trim($randomisationCode, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Log code');
        }

        if(empty($participantIdentifier = mb_trim($participantIdentifier, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Participant identifier');
        }

        if(empty($armAllocated = mb_trim($armAllocated, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Allocated arm');
        }

        if($version < 1) {
            throw new RandomisationException('Version must be greater than 0');
        }

        if(empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Action by');
        }

        $this->study_code = $studyCode;
        $this->randomisation_code = $randomisationCode;
        $this->participant_identifier = $participantIdentifier;
        $this->type = $type;
        $this->arm_allocated = $armAllocated;
        $this->randomised_by = $actionBy;
        $this->randomisation_group = $randomisationGroup;
        $this->run_id = $runId;
        $this->randomised_on = $timestamp ?? new DateTimeImmutable();
        $this->simulation = $simulation;
        $this->simulation_id = $simulationId;
    }

    /**
     * @param string $actionBy
     * @return void
     */
    public function preDelete(string $actionBy): void
    {
        if (empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw new NotBlankException('Actioned by should not be empty');
        }

        $this->deleted_by = $actionBy;
        $this->deleted = new DateTimeImmutable();
    }
}
