<?php

namespace Atlas\RandomisationBundle\Service\Simulator;

use Atlas\RandomisationBundle\Contract\SpecificationInterface;
use Atlas\RandomisationBundle\Entity\Randomisation\Allocation;
use Atlas\RandomisationBundle\Exception\RandomisationException;
use Atlas\RandomisationBundle\Service\Randomisation\Randomizer;
use Atlas\RandomisationBundle\Service\Specification\SpecificationLoader;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Psr\Cache\InvalidArgumentException;
use Symfony\Component\Uid\Uuid;
use Throwable;

readonly class SimulationRunner
{
    public function __construct(
        private Randomizer $randomizer,
        private EntityManagerInterface $entity_manager,
        private SpecificationLoader $loader,
    ) {}

    /**
     * @param string $studyCode
     * @param string $specificationName
     * @param int $version
     * @param array $tests
     * @param string $actionBy
     * @param SpecificationInterface|null $specification
     * @return void
     * @throws InvalidArgumentException
     */
    public function run(
        string $studyCode,
        string $specificationName,
        int $version,
        array $tests,
        string $actionBy,
        ?SpecificationInterface $specification = null,
    ): void
    {

        $specification = $specification ?? $this->loader->load($studyCode, $specificationName, $version);

        $randomisationCode = mb_strtoupper($specification->code, encoding: 'utf-8');
        $studyCode = mb_strtoupper($studyCode, encoding: 'utf-8');

        $results = array();
        $simulate = new DateTimeImmutable();
        $simulateId = Uuid::v7();

        foreach ($tests as $test) {
            try {
                if(! empty($test['manual_arm'])) {
                    $allocation = new Allocation(
                        $studyCode,
                        $randomisationCode,
                        sprintf('test-%s',$test['test_no']),
                        $test['manual_arm'],
                        $version,
                        Uuid::v4(),
                        $actionBy,
                        type: 'manual',
                        simulation: $simulate,
                        simulationId: $simulateId
                    );

                    $this->entity_manager->persist($allocation);
                    $this->entity_manager->flush();

                }
                else {
                    $this->randomizer->randomise(
                        $studyCode,
                        $specificationName,
                        $test['location'] ?? 'Test-001001',
                        $version,
                        sprintf('test-%s',$test['test_no']),
                        $actionBy,
                        variables: $test,
                        simulate: $simulate,
                        simulateId: $simulateId,
                        specification: $specification
                    );
                }
            } catch (RandomisationException $e) {

                $results[] = [
                    'test' => $test,
                    'reason' => $e
                ];
            }
        }

        if (count($results) > 0) {

            $errors = [];
            $i = 1;

            foreach($results as $result) {

                if($result['reason'] instanceof Throwable) {
                    $errors[$result['reason']->getMessage()][] = $i;
                }

                $i++;
            }

            $message = '';

            foreach($errors as $reason => $tests) {

                sort($tests);
                $ranges = [];
                $start = $end = $tests[0];

                foreach (array_slice($tests, 1) as $num) {
                    if ($num === $end + 1) {
                        $end = $num;
                    } else {
                        $ranges[] = [$start, $end];
                        $start = $end = $num;
                    }
                }

                $ranges[] = [$start, $end];

                $parts = array_map(function ($range) {
                    return ($range[0] === $range[1])
                        ? sprintf('Test %d', $range[0])
                        : sprintf('Test %d-%d', $range[0], $range[1]);
                }, $ranges);

                $message .= sprintf('%s: %s; ', implode(', ', $parts), $reason);
            }

            // Either throw here or return it so caller can decide
            throw new RandomisationException($message, 0);
        }
    }
}
