<?php

namespace Atlas\RandomisationBundle\Service\Specification;

use Atlas\RandomisationBundle\Contract\SpecificationInterface;
use Atlas\RandomisationBundle\Exception\SpecificationException;
use Psr\Cache\CacheItemPoolInterface;
use Psr\Cache\InvalidArgumentException;
use Throwable;

readonly class SpecificationCache
{
    public function __construct(
        private SpecificationParser $parser,
        private CacheItemPoolInterface $cache
    )
    {
    }

    /**
     * @param string $studyCode
     * @param string $specificationName
     * @param int $version
     * @param string $path
     * @return SpecificationInterface
     * @throws InvalidArgumentException
     */
    public function load(string $studyCode, string $specificationName, int $version, string $path): SpecificationInterface
    {
        $hash = @hash_file('sha256', $path);

        if (!$hash) {
            throw new SpecificationException(sprintf(
                'Specification for %s %s v%d could not be hashed.',
                $studyCode,
                $specificationName,
                $version
            ));
        }

        $key = sprintf('specification.%s.%s.%d.%s', $studyCode, $specificationName, $version, $hash);

        $key = preg_replace('/[^A-Za-z0-9_.:]/u', '_', $key) ?? $key;

        $cached = $this->cache->getItem($key);
        if ($cached->isHit()) {
            return $cached->get();
        }

        try {

            $specificationArray = (static function (string $__file): array {
                return include $__file;
            })($path);

        } catch (Throwable $e) {
            throw new SpecificationException('Incorrect specification file format.', code: 0, previous: $e);
        }

        $parsed = $this->parser->parse($specificationArray);

        $cached->set($parsed);

        try {
            $saved = $this->cache->save($cached);

            if (!$saved) {
                throw new SpecificationException('Could not save specification cache.');
            }
        } catch (InvalidArgumentException $e) {
            // rethrow with context (or log then rethrow)
            throw new SpecificationException(sprintf('Cache save failed for specification %s: %s', $key, $e->getMessage()), 0, $e);
        }

        return $parsed;
    }
}
