<?php

declare(strict_types=1);

namespace Atlas\RandomisationBundle\Repository\Participant;

use Atlas\RandomisationBundle\Entity\Participant\Factor;
use Atlas\RandomisationBundle\Entity\Randomisation\Allocation;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Bridge\Doctrine\Types\UuidType;
use Symfony\Component\Uid\Uuid;

/**
 * @extends ServiceEntityRepository<Factor>
 */
class FactorRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Factor::class);
    }

    /**
     * Returns counts per factor level and arm — platform-independent.
     */
    public function countsByFactorLevelArm(
        string $studyCode,
        string $randomisationCode,
        ?Uuid $simulateId = null
    ): array {
        $qb = $this->getEntityManager()->createQueryBuilder();

        $qb->select('f.factor AS factor', 'f.value AS level', 'a.arm_allocated AS arm', 'COUNT(a.id) AS n')
            ->from(Factor::class, 'f')
            ->innerJoin(
                Allocation::class,
                'a',
                'WITH',
                'a.study_code = f.study_code AND a.randomisation_code = f.randomisation_code AND a.participant_identifier = f.participant_identifier'
            )
            ->where('f.study_code = :study')
            ->andWhere('f.randomisation_code = :rand')
            ->andWhere('( (a.simulation_id IS NULL AND :sim_id IS NULL) OR (a.simulation_id = :sim_id) )')
            ->groupBy('f.factor, f.value, a.arm_allocated')
            ->orderBy('f.factor', 'ASC')
            ->addOrderBy('f.value', 'ASC')
            ->addOrderBy('a.arm_allocated', 'ASC')
            ->setParameter('study', $studyCode)
            ->setParameter('rand', $randomisationCode)
            ->setParameter('sim_id', $simulateId, UuidType::NAME);

        $rows = $qb->getQuery()->getArrayResult();

        foreach ($rows as &$r) {
            $r['n'] = (int)$r['n'];
        }
        return $rows;
    }

    /**
     * Discovers existing levels for a given factor.
     */
    public function discoverLevels(
        string $studyCode,
        string $randomisationCode,
        string $factorName,
        ?Uuid $simulateId = null
    ): array {
        $qb = $this->getEntityManager()->createQueryBuilder();

        $qb->select('DISTINCT f.value AS level')
            ->from(Factor::class, 'f')
            ->innerJoin(
                Allocation::class,
                'a',
                'WITH',
                'a.study_code = f.study_code AND a.randomisation_code = f.randomisation_code AND a.participant_identifier = f.participant_identifier'
            )
            ->where('f.study_code = :study')
            ->andWhere('f.randomisation_code = :rand')
            ->andWhere('f.factor = :factor')
            ->andWhere('( (a.simulation_id IS NULL AND :sim_id IS NULL) OR (a.simulation_id = :sim_id) )')
            ->orderBy('f.value', 'ASC')
            ->setParameter('study', $studyCode)
            ->setParameter('rand', $randomisationCode)
            ->setParameter('factor', $factorName)
            ->setParameter('sim_id', $simulateId, UuidType::NAME);

        $rows = $qb->getQuery()->getArrayResult();

        return array_values(array_map(fn($r) => (string)$r['level'], $rows));
    }
}