<?php

namespace Atlas\RandomisationBundle\Repository\Study;

use Atlas\RandomisationBundle\Entity\Study\Randomisation;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\AbstractQuery;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @extends ServiceEntityRepository<Randomisation>
 */
class RandomisationRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Randomisation::class);
    }

    /**
     * @param string $code
     * @return Randomisation[]
     */
    public function findByCode(string $code) : array
    {
        $code = mb_trim($code, encoding: 'UTF-8');

        if ($code === '') {
            return [];
        }

        return $this->findBy(['code' => $code]);
    }

    /**
     * @param string $name
     * @param string $studyCode
     * @return Randomisation|null
     */
    public function findBySpecification(string $name, string $studyCode) : ?Randomisation
    {
        $name = mb_trim($name, encoding: 'UTF-8');
        $studyCode = mb_trim($studyCode, encoding: 'UTF-8');

        if ($name === '' || $studyCode === '') {
            return null;
        }

        return $this->findOneBy([
            'specification_name' => $name,
            'study_code' => mb_strtoupper($studyCode, encoding: 'UTF-8')
        ]);
    }

    /**
     * @param string $studyCode
     * @param string $randomisationCode
     * @return string|null
     */
    public function findMasterSpecificationByStudy(string $studyCode, string $randomisationCode): ?string
    {
        $studyCode = mb_trim($studyCode, encoding: 'UTF-8');
        $randomisationCode = mb_trim($randomisationCode, encoding: 'UTF-8');

        if ($studyCode === '') return null;

        return $this->createQueryBuilder('r')
            ->select('DISTINCT r.master_specification')
            ->where('UPPER(r.study_code) = :study_code')
            ->andWhere('UPPER(r.code) = :randomisation_code')
            ->setParameter('randomisation_code', mb_strtoupper($randomisationCode, encoding: 'UTF-8'))
            ->setParameter('study_code', mb_strtoupper($studyCode, encoding: 'UTF-8'))
            ->orderBy('r.master_specification', 'ASC')
            ->getQuery()
            ->getOneOrNullResult(hydrationMode: AbstractQuery::HYDRATE_SINGLE_SCALAR);
    }

    /**
     * @param string $studyCode
     * @return string[]
     */
    public function findByStudyCode(string $studyCode): array
    {
        $studyCode = mb_trim($studyCode, encoding: 'UTF-8');

        if ($studyCode === '') return [];

        return array_column($this->createQueryBuilder('r')
            ->select('DISTINCT r.code')
            ->where('UPPER(r.study_code) = :studyCode')
            ->setParameter('studyCode', mb_strtoupper($studyCode, encoding: 'UTF-8'))
            ->orderBy('r.code', 'ASC')
            ->getQuery()
            ->getScalarResult(), 'code');
    }
}
