<?php

namespace Atlas\RandomisationBundle\Service\Specification;

use Atlas\RandomisationBundle\Contract\SpecificationInterface;
use Atlas\RandomisationBundle\Contract\SpecificationParserInterface;
use Atlas\RandomisationBundle\Exception\SpecificationException;
use Atlas\RandomisationBundle\Service\Specification\Traits\KeyCheckerTrait;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\DependencyInjection\Attribute\AutowireLocator;
use Symfony\Contracts\Service\ServiceProviderInterface;

final readonly class SpecificationParser
{

    use KeyCheckerTrait;

    private const array SYSTEM_KEYS = [
        'code',
        'system',
        'when',
        'randomisation' => [
            'type',
            'arms'
        ]
    ];

    public function __construct(
        #[Autowire(param: 'shared.randomisation.algorithms')]
        private array $enabled_randomisation,
        #[AutowireLocator('atlas.randomisation.parser', indexAttribute: 'algorithm')]
        private ServiceProviderInterface $parsers
    )
    {}

    public function parse(array $specification): SpecificationInterface
    {
        $this->assertKeys($specification, self::SYSTEM_KEYS, required: true);

        $type = $specification['randomisation']['type'] ?? null;

        if (! is_string($type) || $type === '') {
            throw new SpecificationException('randomisation.type is required and must be a non-empty string.');
        }

        $parser = $this->getParser($type);

        if($parser === null) {
            throw new SpecificationException(sprintf(
                'Cannot find parser for type "%s"',
                $type
            ));
        }

        return $parser->parse($specification);
    }

    private function getParser(string $algorithm): ?SpecificationParserInterface
    {
        $algorithm = mb_trim(mb_strtolower($algorithm, encoding: 'UTF-8'), encoding: 'UTF-8');

        if($algorithm === '') {
            return null;
        }

        if(! in_array($algorithm, $this->enabled_randomisation, true)) return null;

        if(! $this->parsers->has($algorithm)) {
            return null;
        }

        return $this->parsers->get($algorithm);
    }

}