<?php

namespace Atlas\RandomisationBundle\Controller;


use Atlas\RandomisationBundle\Exception\RandomisationException;
use Atlas\RandomisationBundle\Exception\SpecificationException;
use Atlas\RandomisationBundle\Security\RandomisationAuthorisationInterface;
use Atlas\RandomisationBundle\Service\Simulator\SimulationRunner;
use Atlas\RandomisationBundle\Service\Specification\SpecificationLoader;
use Atlas\RandomisationBundle\Service\Spreadsheet\ImporterService as SpreadsheetImporter;
use Psr\Cache\InvalidArgumentException;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Cache\Exception\CacheException;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Throwable;

#[IsGranted('ROLE_USER')]
final class SimulateController extends AbstractController
{

    private const array EXT  = ['xlsx', 'xls', 'csv'];
    private const array MIME  = [
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        'application/vnd.ms-excel',
        'text/csv',
        'application/csv',
        'text/plain',
        'application/octet-stream',
    ];

    public function __construct(
        private readonly RandomisationAuthorisationInterface $authorisation
    )
    {}

    #[Route(
        '/simulate/randomisation/study/{study<[A-Za-z0-9_-]+>}',
        name: 'shared_rand_simulate',
        defaults: ['_uppercase' => ['code', 'study']],
        methods: ['GET'])
    ]
    public function index(
        string $study
    ): Response {

        if (!$this->authorisation->canSimulate($study)) {
            throw new AccessDeniedHttpException();
        }

        if ($this->authorisation->isBlinded($study)) {
            throw new AccessDeniedHttpException();
        }

        return $this->render('@Randomisation/simulate/index.html.twig', [
            'study' => $study
        ]);
    }

    /**
     * @throws InvalidArgumentException
     */
    #[Route(
        '/simulate/randomisation/study/{study<[A-Za-z0-9_-]+>}',
        name: 'shared_rand_simulate_upload',
        defaults: ['_uppercase' => ['code', 'study']],
        methods: ['POST'])
    ]
    public function upload(
        Request $request,
        string $study,
        SpecificationLoader $loader,
        SpreadsheetImporter $importer,
        SimulationRunner $runner
    ): Response {

        if (!$this->authorisation->canSimulate($study)) {
            throw new AccessDeniedHttpException();
        }

        if ($this->authorisation->isBlinded($study)) {
            throw new AccessDeniedHttpException();
        }

        if (!$this->isCsrfTokenValid('simulate_upload_'.$study, (string)$request->request->get('_token'))) {
            return $this->errorRedirect(
                'Invalid session token.',
                [
                    'study' => $study
                ]
            );
        }

        $file = $request->files->get('spreadsheet');

        if (!$file instanceof UploadedFile) {
            return $this->errorRedirect(
                'No file uploaded.',
                [
                    'study' => $study
                ]
            );
        }

        $ext  = mb_strtolower($file->getClientOriginalExtension(), encoding: 'UTF-8');
        $mime = (string) $file->getMimeType();

        if (!in_array($ext, self::EXT, true)) {
            return $this->errorRedirect(
                'Unsupported file type.',
                [
                    'study' => $study
                ]
            );
        }
        if (!in_array($mime, self::MIME, true)) {
            return $this->errorRedirect(
                'Unsupported MIME type.',
                [
                    'study' => $study
                ]
            );
        }

        if ($file->getSize() !== null && $file->getSize() > 20 * 1024 * 1024) {
            return $this->errorRedirect(
                'File too large (max 20 MB).',
                [
                    'study' => $study
                ]
            );
        }

        $specificationName = trim((string) $request->request->get('specification', ''));
        $version = $request->request->getInt('version', -1);

        if (empty($specificationName) || $version < 0) {
            return $this->errorRedirect(
                'Specification is required and version must be a non-negative integer.',
                [
                    'study' => $study
                ]
            );
        }

        $user = $this->getUser();

        try {
            // Import spreadsheet rows
            $rows = $importer->import($file);

            // Run simulation
            $runner->run($study, $specificationName, $version, $rows, $user->getUserIdentifier());

        } catch (SpecificationException $e) {
            return $this->errorRedirect(
                $e->getMessage(),
                [
                    'study' => $study
                ]
            );

        } catch (CacheException) {
            return $this->errorRedirect(
                'Cannot access specification from cache.',
                [
                    'study' => $study
                ]
            );

        } catch (RandomisationException $e) {

            $this->handleRandomisationFailure($e);

            return $this->redirectToRoute(
                'shared_rand_simulate',
                [
                    'study' => $study
                ]
            );

        }

        $this->addFlash('success', 'File processed successfully! You can now download it.');
        return $this->redirectToRoute(
            'shared_rand_simulate',
            [
                'study' => $study
            ]
        );
    }

    /**
     * @param string $message
     * @param array $params
     * @return Response
     */
    private function errorRedirect(string $message, array $params = []): Response
    {
        $this->addFlash('error', $message);
        return $this->redirectToRoute('shared_rand_simulate', $params);
    }

    /**
     * @param RandomisationException $e
     * @return void
     */
    private function handleRandomisationFailure(RandomisationException $e): void
    {

        // Detailed error messages for each failed test
        $results = method_exists($e, 'getResults') ? $e->getResults() : [];

        if (!empty($results)) {
            foreach ($results as $failure) {
                $reason = $failure['reason'];
                $message = sprintf(
                    'Test "%s" failed: %s',
                    $failure['test'],
                    $reason instanceof Throwable ? $reason->getMessage() : (string)$reason
                );
                $this->addFlash('error', $message);
            }
        } else {
            $this->addFlash('error', $e->getMessage());
        }
    }
}
