<?php

declare(strict_types=1);

namespace Atlas\RandomisationBundle\Service\Randomisation;

use Atlas\RandomisationBundle\Contract\SpecificationInterface;
use Atlas\RandomisationBundle\Domain\Arm;
use Atlas\RandomisationBundle\Dto\ParticipantAllocationDto;
use Atlas\RandomisationBundle\Repository\Randomisation\AllocationRepository;
use Atlas\RandomisationBundle\Service\Specification\SpecificationLoader;
use DateTimeImmutable;
use DateTimeInterface;
use Psr\Cache\InvalidArgumentException;

final readonly class AllocationManager
{
    public function __construct(
        private AllocationRepository $allocations,
        private SpecificationLoader $specification_loader,
    ) {}

    /**
     * @return array<int, ParticipantAllocationDto>
     * @throws InvalidArgumentException
     * @throws \DateMalformedStringException
     */
    public function getParticipantAllocations(string $studyCode, string $participantIdentifier, bool $blinded = true): array
    {
        $rows = $this->allocations->findParticipantsAllocations($studyCode, $participantIdentifier);

        if ($rows === []) {
            return [];
        }

        /** @var array<string, SpecificationInterface> $specCache */
        $specCache = [];

        $dtos = [];

        foreach ($rows as $row) {
            $specName = $row['specification_name'];
            $version = (int) $row['version'];

            $cacheKey = $specName . '#v' . $version;

            $spec = $specCache[$cacheKey]
                ??= $this->specification_loader->load($studyCode, $specName, $version);

            $allocatedCode = (string) $row['allocation_code'];

            $randomised = $row['randomised'];
            if (is_string($randomised)) {
                $randomised = new DateTimeImmutable($randomised);
            }

            /** @var DateTimeInterface $randomised */
            $dtos[] = new ParticipantAllocationDto(
                randomisation_code: $row['randomisation_code'],
                randomisation_name: $row['randomisation_name'],
                randomisation_group: $row['randomisation_group'],
                type: $row['type'],
                allocated_code: $blinded ? 'BLIND' : $allocatedCode,
                allocated_name: $blinded ? 'Completed' : $this->resolveAllocatedName($spec->arms, $allocatedCode),
                randomised: $randomised,
                randomised_by: $row['randomised_by'],
            );
        }

        return $dtos;
    }

    /**
     * @param array<string, Arm> $arms
     */
    private function resolveAllocatedName(array $arms, string $allocatedCode): string
    {
        $allocatedCode = mb_trim($allocatedCode, encoding: 'UTF-8');

        // Your spec builder makes the map key the arm CODE (e.g. PLC/INT).
        // Arm::$arm also holds that code.
        if (isset($arms[$allocatedCode])) {
            return $arms[$allocatedCode]->arm;
        }

        // Fallback: match against arm code or export.
        foreach ($arms as $arm) {
            if ($arm->arm === $allocatedCode) {
                return $arm->arm;
            }

            if ($arm->export !== null && (string) $arm->export === $allocatedCode) {
                return $arm->arm;
            }
        }

        return $allocatedCode;
    }
}