<?php

declare(strict_types=1);

namespace Atlas\RandomisationBundle\Service\Report;

use Atlas\RandomisationBundle\Contract\ReportInterface;
use Atlas\RandomisationBundle\Exception\ReportException;
use Atlas\RandomisationBundle\Repository\Randomisation\AllocationRepository;
use Atlas\RandomisationBundle\Repository\Study\RandomisationRepository;
use Atlas\RandomisationBundle\Service\Specification\SpecificationLoader;
use Atlas\ReportRenderBundle\Model\Chart;
use Atlas\ReportRenderBundle\Model\Layout;
use Atlas\ReportRenderBundle\Model\SectionInterface;
use Atlas\ReportRenderBundle\Model\Table;
use Psr\Cache\InvalidArgumentException;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\DependencyInjection\Attribute\AutowireLocator;
use Symfony\Component\Uid\Uuid;
use Symfony\Contracts\Service\ServiceProviderInterface;
use Symfony\UX\Chartjs\Builder\ChartBuilderInterface;
use Symfony\UX\Chartjs\Model\Chart as ChartJs;

final readonly class ReportBuilder
{
    /**
     * @param array $enabled_randomisation
     * @param ServiceProviderInterface $reporters
     * @param AllocationRepository $allocations
     * @param RandomisationRepository $randomisation
     * @param SpecificationLoader $specification_loader
     * @param ChartBuilderInterface $charts
     */
    public function __construct(
        #[Autowire(param: 'shared.randomisation.algorithms')]
        private array $enabled_randomisation,
        #[AutowireLocator('atlas.randomisation.reporter', indexAttribute: 'algorithm')]
        private ServiceProviderInterface $reporters,
        private AllocationRepository $allocations,
        private RandomisationRepository $randomisation,
        private SpecificationLoader $specification_loader,
        private ChartBuilderInterface $charts
    )
    {}

    /**
     * @param string $studyCode
     * @param string $randomisationCode
     * @param Uuid|null $simulated
     * @return Layout
     * @throws InvalidArgumentException
     */
    public function generate(
        string $studyCode,
        string $randomisationCode,
        ?Uuid $simulated = null
    ): Layout
    {
        $master = $this->randomisation->findMasterSpecificationByStudy($studyCode, $randomisationCode);

        $specification = $this->specification_loader->load($studyCode, $master);

        $reporter = $this->getAlgorithm($specification->getType());

        if($reporter === null) {
            throw new ReportException(sprintf(
                'Cannot find reporter for type "%s" (code: %s, study: %s)',
                $specification->getType(), $randomisationCode, $studyCode
            ));
        }

        $base = $this->buildAllocationSections($studyCode, $randomisationCode, simulated: $simulated);

        $generated = $reporter->buildAlgorithmSections($studyCode, $randomisationCode, specification: $specification, simulated: $simulated);

        $sections = array_merge($base, $generated);

        $title = sprintf('Log Report for Study Code: %s and Log Code: %s', $studyCode, $randomisationCode);
        if($simulated !== null) $title = sprintf('%s for Simulation: %s', $title, $simulated->toRfc4122());

        return new Layout(
            $title,
            $sections
        );
    }

    private function getAlgorithm(string $algorithm): ?ReportInterface
    {
        $algorithm = mb_trim(mb_strtolower($algorithm, encoding: 'UTF-8'), encoding: 'UTF-8');

        if($algorithm === '') return null;

        if(! in_array($algorithm, $this->enabled_randomisation, true)) return null;

        if(! $this->reporters->has($algorithm)) return null;

        return $this->reporters->get($algorithm);
    }

    /**
     * @param string $studyCode
     * @param string $randomisationCode
     * @param Uuid|null $simulated
     * @return SectionInterface[]
     */
    private function buildAllocationSections(
        string $studyCode,
        string $randomisationCode,
        ?Uuid $simulated
    ): array
    {
        $sections = [];

        $counts = $this->allocations->countAllocationsByArm($studyCode, $randomisationCode, simulationId: $simulated);

        $table = [];
        $total = array_sum($counts);

        $labels = [];
        $data = [];

        foreach ($counts as $key => $count) {
            $labels[] = $key;
            $data[] = $count;

            $percentage = $total > 0 ? $count / $total * 100 : 0;

            $table[] = [
                'arm' => $key,
                'count' =>  sprintf('%.2f%% (n=%d)', $percentage, $count)
            ];
        }

        $graph = $this->charts->createChart(ChartJs::TYPE_DOUGHNUT);

        $graph->setOptions([
            'responsive' => true,
            'plugins' => [
                'legend' => ['position' => 'bottom'],
                'tooltip' => ['enabled' => true],
            ],
        ]);

        $colours = Chart::getColours(count($data));

        $graph->setData(
            [
                'labels' => $labels, // The X-axis labels (Arm A, Arm B, etc.)
                'datasets' => [
                    [
                        'label' => 'Total Participants Allocated',
                        'data' => $data,
                        'backgroundColor' => $colours
                    ],
                ],
            ]
        );

        $sections[] = new Chart('Allocations', $graph);

        $headings = [
            'arm' => 'Arms',
            'count' => 'Count'
        ];

        $sections[] = new Table($headings, $table, title: sprintf('Allocations by Arms: %d', $total));

        return $sections;
    }
}
