<?php

namespace Atlas\RandomisationBundle\Service\Specification;

use Atlas\RandomisationBundle\Contract\SpecificationInterface;
use Atlas\RandomisationBundle\Exception\SpecificationException;
use Psr\Cache\InvalidArgumentException;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

readonly class SpecificationLoader
{
    public function __construct(
        #[Autowire(param: 'shared.randomisation.spec_root')]
        private string $base_dir,
        private SpecificationCache $cache
    ) {}

    /**
     * @param string $studyCode
     * @param string $specificationName
     * @param ?int $version
     * @return SpecificationInterface
     * @throws InvalidArgumentException
     */
    public function load(string $studyCode, string $specificationName, ?int $version = null): SpecificationInterface
    {
        if($version === null) {
            [ $version, $path ] = $this->getVersion($studyCode, $specificationName);
        }
        else {
            $path = $this->getPath($studyCode, $specificationName, $version);
        }

        return $this->cache->load($studyCode, $specificationName, $version, $path);
    }

    /**
     * @param string $studyCode
     * @param string $specificationName
     * @param int $version
     * @return string
     */
    private function getPath(string $studyCode, string $specificationName, int $version): string
    {
        $dir = rtrim($this->base_dir, '/');

        $candidates = [
            sprintf('%s/%s/v%d/%s.php', $dir, strtoupper($studyCode), $version, strtoupper($specificationName)),
            sprintf('%s/%s/v%d/%s.php', $dir, strtolower($studyCode), $version, strtolower($specificationName)),
        ];

        $errors = [];

        foreach ($candidates as $path) {
            if (!is_file($path)) {
                $errors[] = sprintf('Not found: %s', $path);
                continue;
            }
            if (!is_readable($path)) {
                $errors[] = sprintf('Not readable: %s', $path);
                continue;
            }
            return $path;
        }

        throw new SpecificationException(sprintf(
            'Specification for %s %s v%d could not be located: %s',
            $studyCode,
            $specificationName,
            $version,
            implode('; ', $errors)
        ));
    }

    /**
     * @param string $studyCode
     * @param string $specificationName
     * @return array
     */
    private function getVersion(string $studyCode, string $specificationName): array
    {
        $dir = rtrim($this->base_dir, '/');

        $candidates = [
            sprintf('%s/%s', $dir, strtoupper($studyCode)),
            sprintf('%s/%s', $dir, strtolower($studyCode)),
        ];

        $versions = [];
        foreach ($candidates as $candidate) {
            if (!is_dir($candidate)) {
                continue;
            }

            $found = glob($candidate . '/v[0-9]*', GLOB_ONLYDIR);
            if ($found === false) {
                continue; // glob error
            }

            foreach ($found as $directory) {
                $version = ltrim(basename($directory), 'v');
                // Ensure it's only digits (e.g., not "v1-beta")
                if (ctype_digit($version)) {
                    $versions[(int) $version] = true; // Use keys for automatic uniqueness
                }
            }
        }

        $versions = array_keys($versions);

        if (empty($versions)) {
            throw new SpecificationException(sprintf(
                'No version directories (e.g., "v1", "v2") found for study %s',
                $studyCode
            ));
        }

        rsort($versions, flags: SORT_NUMERIC);

        foreach ($versions as $version) {
            try {
                return [ $version, $this->getPath($studyCode, $specificationName, $version) ];
            }
            catch(SpecificationException) {
                //do nothing
            }
        }

        throw new SpecificationException(sprintf(
            'No version directories (e.g., "v1", "v2") found for study %s and randomisation %s',
            $studyCode,
            $specificationName
        ));
    }
}