<?php

namespace Atlas\RandomisationBundle\Service\Export;

use Atlas\ExportBundle\Service\Exporter;
use Atlas\RandomisationBundle\Contract\ExportInterface;
use Atlas\RandomisationBundle\Contract\SpecificationInterface;
use Atlas\RandomisationBundle\Dto\ExportTypeDto;
use Atlas\RandomisationBundle\Exception\ExportException;
use Atlas\RandomisationBundle\Repository\Audit\RandomisationRepository;
use Atlas\RandomisationBundle\Repository\Participant\FactorRepository;
use Atlas\RandomisationBundle\Repository\Randomisation\AllocationRepository;
use Atlas\RandomisationBundle\Service\Specification\SpecificationLoader;
use Psr\Cache\InvalidArgumentException;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\DependencyInjection\Attribute\AutowireLocator;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Uid\Uuid;
use Symfony\Contracts\Service\ServiceProviderInterface;

final readonly class ExportGenerator
{

    public function __construct(
        #[Autowire(param: 'shared.randomisation.algorithms')]
        private array $enabled_randomisation,
        #[AutowireLocator('atlas.randomisation.exporter', indexAttribute: 'algorithm')]
        private ServiceProviderInterface $exporters,
        private SpecificationLoader $specification_loader,
        private Exporter $exporter,
        private RandomisationRepository $logs,
        private AllocationRepository $allocations
    )
    {}

    /**
     * @param string $studyCode
     * @param string $randomisationCode
     * @param string $specificationName
     * @param SpecificationInterface|null $specification
     * @return ExportTypeDto[]
     * @throws InvalidArgumentException
     */
    public function list(
        string  $studyCode,
        string  $randomisationCode,
        string  $specificationName,
        ?SpecificationInterface $specification = null
    ): array
    {

        $specification = $specification ?? $this->specification_loader->load($studyCode, $specificationName);

        $exporter = $this->getAlgorithm($specification->getType());

        if($exporter === null) {
            throw new ExportException(sprintf(
                'Cannot find exporter for type "%s" (code: %s, study: %s)',
                $specification->getType(),
                $randomisationCode,
                $studyCode
            ));
        }

        $default = [
            new ExportTypeDto('Allocations', 'allocation'),
            new ExportTypeDto('Randomisation Logs', 'log'),
        ];

        return array_merge($default, $exporter->getAvailableExports());
    }

    /**
     * @param string $studyCode
     * @param string $randomisationCode
     * @param string $specificationName
     * @param string|null $type
     * @param string|null $method
     * @param Uuid|null $simulated
     * @param SpecificationInterface|null $specification
     * @return Response
     * @throws InvalidArgumentException
     */
    public function export(
        string  $studyCode,
        string  $randomisationCode,
        string  $specificationName,
        ?string  $type = null,
        ?string  $method = null,
        ?Uuid $simulated = null,
        ?SpecificationInterface $specification = null
    ): Response {

        $specification = $specification ?? $this->specification_loader->load($studyCode, $specificationName);

        $exporter = $this->getAlgorithm($specification->getType());

        if($exporter === null) {
            throw new ExportException(sprintf(
                'Cannot find exporter for type "%s" (code: %s, study: %s)',
                $specification->getType(), $randomisationCode, $studyCode
            ));
        }

        if($type === null || $type === 'allocation' || $type === 'log') {

            if($type === 'log') {
                $data = $this->normaliseLogs($this->logs->getLogs($studyCode, $randomisationCode, simulated: $simulated));
            }
            else{
                $data = $this->normaliseAllocations($this->allocations->getAllocations($studyCode, $randomisationCode, simulated: $simulated));
            }

            return match($method) {
                ExportInterface::CSV => $this->exporter->csv($data),
                ExportInterface::EXCEL => $this->exporter->excel($data),
                ExportInterface::SPSS => $this->exporter->spss($data),
                default => throw new ExportException(sprintf(
                    'Unsupported export method "%s" for "%s" export',
                    $method,
                    $type ?? 'allocation'
                ))
            };
        }

        return $exporter->export($studyCode, $randomisationCode,$type, $method, simulated: $simulated);
    }

    private function getAlgorithm(string $algorithm): ?ExportInterface
    {
        $algorithm = mb_trim(mb_strtolower($algorithm, encoding: 'UTF-8'), encoding: 'UTF-8');

        if($algorithm === '') return null;

        if(! in_array($algorithm, $this->enabled_randomisation, true)) return null;

        if(! $this->exporters->has($algorithm)) return null;

        return $this->exporters->get($algorithm);
    }

    private function normaliseAllocations(array $data): array
    {
        return array_map(
            static function ($allocation): array {
                return [
                    'run_id' => $allocation->run_id->toRfc4122(),
                    'participant_identifier' => $allocation->participant_identifier,
                    'randomisation_group' => $allocation->randomisation_group,
                    'type' => $allocation->type,
                    'arm_allocated' => $allocation->arm_allocated,
                    'randomised_on' => $allocation->randomised_on->format('c'),
                    'randomised_by' => $allocation->randomised_by,
                ];
            },
            $data
        );
    }

    private function normaliseLogs(array $log): array
    {
        return array_map(
            static function ($log): array {
                return [
                    'run_id' => $log->run_id->toRfc4122(),
                    'participant_identifier' => $log->participant_identifier,
                    'message' => $log->message,
                    'action_by' => $log->action_by,
                    'timestamp' => $log->timestamp->format('c')
                ];
            },
            $log
        );
    }
}
