<?php

declare(strict_types=1);

namespace Atlas\RandomisationBundle\Service\Randomisation;

use Atlas\RandomisationBundle\Contract\SpecificationInterface;
use Atlas\RandomisationBundle\Domain\Arm;
use Atlas\RandomisationBundle\Dto\ParticipantAllocationDto;
use Atlas\RandomisationBundle\Repository\Randomisation\AllocationRepository;
use Atlas\RandomisationBundle\Service\Specification\SpecificationLoader;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\ORM\NonUniqueResultException;
use Psr\Cache\InvalidArgumentException;

final readonly class AllocationManager
{
    public function __construct(
        private AllocationRepository $allocations,
        private SpecificationLoader $specification_loader,
    ) {}

    /**
     * @return array<int, ParticipantAllocationDto>
     * @throws InvalidArgumentException
     * @throws \DateMalformedStringException
     */
    public function getParticipantAllocations(string $studyCode, string $participantIdentifier, bool $blinded = true): array
    {
        $rows = $this->allocations->findParticipantsAllocations($studyCode, $participantIdentifier);

        if ($rows === []) {
            return [];
        }

        /** @var array<string, SpecificationInterface> $specCache */
        $specCache = [];

        $dtos = [];

        foreach ($rows as $row) {
            $specName = $row['specification_name'];
            $version = (int) $row['version'];

            $cacheKey = $specName . '#v' . $version;

            $spec = $specCache[$cacheKey]
                ??= $this->specification_loader->load($studyCode, $specName, $version);

            $allocatedCode = (string) $row['allocation_code'];
            $allocatedName = $spec->arms[$allocatedCode]?->export ?? $allocatedCode;

            $randomised = $row['randomised'];
            if (is_string($randomised)) {
                $randomised = new DateTimeImmutable($randomised);
            }

            /** @var DateTimeInterface $randomised */
            $dtos[] = new ParticipantAllocationDto(
                randomisation_code: $row['randomisation_code'],
                randomisation_name: $row['randomisation_name'],
                randomisation_group: $row['randomisation_group'],
                type: $row['type'],
                allocated_code: $blinded ? 'BLIND' : $allocatedCode,
                allocated_name: $blinded ? 'Completed' : $allocatedName,
                randomised: $randomised,
                randomised_by: $row['randomised_by'],
            );
        }

        return $dtos;
    }

    /**
     * Get a participant allocation for a specific randomisation.
     *
     * @throws InvalidArgumentException
     * @throws \DateMalformedStringException
     * @throws NonUniqueResultException
     */
    public function getParticipantAllocation(
        string $studyCode,
        string $participantIdentifier,
        string $randomisationCode,
        bool $blinded = true,
    ): ?ParticipantAllocationDto {
        $row = $this->allocations->findParticipantAllocationForRandomisation(
            $studyCode,
            $participantIdentifier,
            $randomisationCode,
        );

        if ($row === null) {
            return null;
        }

        $specName = $row['specification_name'];
        $version = (int) $row['version'];

        $spec = $this->specification_loader->load($studyCode, $specName, $version);

        $allocatedCode = (string) $row['allocation_code'];
        $allocatedName = $spec->arms[$allocatedCode]?->export ?? $allocatedCode;

        $randomised = $row['randomised'];
        if (is_string($randomised)) {
            $randomised = new DateTimeImmutable($randomised);
        }

        /** @var DateTimeInterface $randomised */
        return new ParticipantAllocationDto(
            randomisation_code: $row['randomisation_code'],
            randomisation_name: $row['randomisation_name'],
            randomisation_group: $row['randomisation_group'],
            type: $row['type'],
            allocated_code: $blinded ? 'BLIND' : $allocatedCode,
            allocated_name: $blinded ? 'Completed' : $allocatedName,
            randomised: $randomised,
            randomised_by: $row['randomised_by'],
        );
    }
}