<?php

declare(strict_types=1);

namespace Atlas\RandomisationBundle\Entity\Study;

use Atlas\AuditBundle\Attribute\AuditActor;
use Atlas\AuditBundle\Attribute\AuditTimestamp;
use Atlas\AuditBundle\Attribute\Enum\AuditActionType;
use Atlas\AuditBundle\Attribute\NotLogged;
use Atlas\RandomisationBundle\Repository\Study\RandomisationRepository;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Validator\Constraints\Length;
use Symfony\Component\Validator\Constraints\NotBlank;

#[ORM\Entity(repositoryClass: RandomisationRepository::class)]
#[ORM\Table(name: 'sys_randomisation')]
#[ORM\UniqueConstraint(columns: ['study_code', 'code', 'specification_name'])]
#[UniqueEntity(fields: ['study_code', 'code', 'specification_name'])]
class Randomisation
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\Column(length: 10)]
    #[Length(min: 3, max: 10)]
    private(set) string $code;

    #[ORM\Column]
    private(set) string $name;

    #[ORM\Column]
    private(set) string $specification_name;

    #[ORM\Column]
    private(set) string $master_specification;

    #[ORM\Column]
    private(set) string $study_code;

    #[NotLogged]
    #[ORM\Column(type: Types::DATETIME_MUTABLE)]
    #[AuditTimestamp(AuditActionType::INSERT)]
    private(set) DateTimeInterface $created;

    #[NotLogged]
    #[ORM\Column(length: 255)]
    #[AuditActor(AuditActionType::INSERT)]
    private(set) string $created_by;

    #[NotLogged]
    #[ORM\Column(type: Types::DATETIME_MUTABLE)]
    #[AuditTimestamp(AuditActionType::UPDATE)]
    private(set) DateTimeInterface $modified;

    #[NotLogged]
    #[ORM\Column(length: 255)]
    #[AuditActor(AuditActionType::INSERT)]
    private(set) string $modified_by;

    #[NotLogged]
    #[ORM\Column(length: 255)]
    #[Length(min: 10, max: 255)]
    #[NotBlank]
    private(set) string $reason;

    /**
     * @param string $code
     * @param string $name
     * @param string $specificationName
     * @param string $studyCode
     * @param string $masterSpecification
     * @param string $actionBy
     * @param DateTimeInterface|null $timestamp
     */
    public function __construct(
        string $code,
        string $name,
        string $specificationName,
        string $studyCode,
        string $masterSpecification,
        string $actionBy,
        ?DateTimeInterface $timestamp = null
    )
    {
        $this->code = $code;
        $this->name = $name;
        $this->specification_name = $specificationName;
        $this->study_code = $studyCode;
        $this->master_specification = $masterSpecification;
        $this->created_by = $this->modified_by = $actionBy;
        $this->created = $this->modified = $timestamp ?? new DateTimeImmutable();
        $this->reason = 'Initial creation of randomisation';
    }
}
