<?php

namespace Atlas\RandomisationBundle\Entity\Participant;

use Atlas\AuditBundle\Attribute\AuditActor;
use Atlas\AuditBundle\Attribute\AuditTimestamp;
use Atlas\AuditBundle\Attribute\Enum\AuditActionType;
use Atlas\AuditBundle\Attribute\NotLogged;
use Atlas\RandomisationBundle\Exception\NotBlankException;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Types\UuidType;
use Symfony\Component\Uid\Uuid;
use Symfony\Component\Validator\Constraints\NotBlank;

#[ORM\Entity]
#[ORM\Table(name: 'rand_factor')]
#[ORM\Index(columns: ['study_code'])]
#[ORM\Index(columns: ['randomisation_code'])]
#[ORM\Index(columns: ['participant_identifier'])]
#[ORM\Index(columns: ['factor'])]
#[ORM\Index(columns: ['run_id'])]
#[ORM\Index(columns: ['simulation_id'])]
#[ORM\UniqueConstraint(columns: ["study_code","randomisation_code","participant_identifier","factor","simulation_id"])]
class Factor
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\Column]
    public string $study_code;

    #[ORM\Column]
    private(set) string $randomisation_code;

    #[ORM\Column]
    private(set) string $participant_identifier;

    #[ORM\Column]
    public string $factor;

    #[ORM\Column]
    public string $value;

    #[ORM\Column(type: UuidType::NAME)]
    private(set) ?Uuid $run_id = null;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[NotLogged]
    #[AuditTimestamp(AuditActionType::INSERT)]
    private(set) DateTimeInterface $created;

    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[NotLogged]
    #[AuditActor(AuditActionType::INSERT)]
    private(set) string $created_by;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[NotLogged]
    #[AuditTimestamp(AuditActionType::UPDATE)]
    private(set) DateTimeInterface $modified;

    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[NotLogged]
    #[AuditActor(AuditActionType::UPDATE)]
    private(set) string $modified_by;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE, nullable: true, options: ['default' => null])]
    private(set) ?DateTimeInterface $simulation = null;

    #[ORM\Column(type: UuidType::NAME, nullable: true, options: ['default' => null])]
    private(set) ?Uuid $simulation_id = null;

    /**
     * @param string $studyCode
     * @param string $randomisationCode
     * @param string $participantIdentifier
     * @param string $factor
     * @param string $value
     * @param Uuid $runId
     * @param string $actionBy
     * @param DateTimeInterface|null $timestamp
     * @param DateTimeInterface|null $simulation
     * @param Uuid|null $simulationId
     */
    public function __construct(
        string $studyCode,
        string $randomisationCode,
        string $participantIdentifier,
        string $factor,
        string $value,
        Uuid $runId,
        string $actionBy,
        ?DateTimeInterface $timestamp = null,
        ?DateTimeInterface $simulation = null,
        ?Uuid $simulationId = null
    )
    {
        if(empty($studyCode = mb_trim($studyCode, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Study code');
        }

        if(empty($randomisationCode = mb_trim($randomisationCode, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Log code');
        }

        if(empty($participantIdentifier = mb_trim($participantIdentifier, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Participant identifier');
        }

        if(empty($factor = mb_trim($factor, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Factor');
        }

        if(empty($value = mb_trim($value, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Value');
        }

        if(empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Action by');
        }

        $this->study_code = $studyCode;
        $this->randomisation_code = $randomisationCode;
        $this->participant_identifier = $participantIdentifier;
        $this->factor = $factor;
        $this->value = $value;
        $this->created_by = $this->modified_by = $actionBy;
        $this->run_id = $runId;
        $this->created = $this->modified = $timestamp ?? new DateTimeImmutable();
        $this->simulation = $simulation;
        $this->simulation_id = $simulationId;
    }

}
