<?php

namespace Atlas\RandomisationBundle\Service\Specification\Traits;

use Atlas\RandomisationBundle\Exception\SpecificationException;

trait KeyCheckerTrait
{
    protected function assertBadKeys(array $specification, array $keys): void
    {
        $this->assertKeys($specification, $keys, required: false);
    }

    protected function assertRequiredAlgorithmKeys(array $specification, array $keys): void
    {
        $this->assertKeys($specification, $keys, required: true);
    }


    /**
     * Unified validator for required/disallowed keys.
     *
     * @param array $spec The spec being validated
     * @param array $schema Shape definition (bare keys or sections=>children)
     * @param bool $required true = assert presence; false = assert absence
     * @param string|null $context Optional context text to append to messages (e.g., "for simple randomisation")
     */
    private function assertKeys(array $spec, array $schema, bool $required, ?string $context = null): void
    {
        $suffix = $context ? " $context" : '';

        foreach ($schema as $section => $config) {
            // Bare top-level key: numeric index -> value is the key name
            if (is_int($section)) {
                $key = (string)$config;

                if ($required) {
                    if (!array_key_exists($key, $spec)) {
                        throw new SpecificationException(sprintf("Key %s required%s", $key, $suffix));
                    }
                } else {
                    if (array_key_exists($key, $spec)) {
                        throw new SpecificationException(sprintf("Key %s not allowed%s", $key, $suffix));
                    }
                }
                continue;
            }

            // Section with optional children list
            $hasSection = array_key_exists($section, $spec);

            if ($required) {
                if (!$hasSection) {
                    throw new SpecificationException(sprintf("Key %s required%s", $section, $suffix));
                }
                // If children are specified, validate each
                if (is_array($config) && $config !== []) {
                    if (!is_array($spec[$section])) {
                        throw new SpecificationException(sprintf("Key %s required%s", $section, $suffix));
                    }
                    foreach ($config as $child) {
                        if (!array_key_exists($child, $spec[$section])) {
                            throw new SpecificationException(sprintf("Key %s.%s required%s", $section, $child, $suffix));
                        }
                    }
                }
            } else {
                // Disallowed
                if (!$hasSection) {
                    continue; // nothing to do
                }
                if (is_array($config) && $config !== []) {
                    // Only these children are disallowed within the section
                    if (is_array($spec[$section])) {
                        foreach ($config as $child) {
                            if (array_key_exists($child, $spec[$section])) {
                                throw new SpecificationException(sprintf("Key %s.%s not allowed%s", $section, $child, $suffix));
                            }
                        }
                    } else {
                        // Section exists but isn't an array: treat presence as disallowed (entire section)
                        throw new SpecificationException(sprintf("Key %s not allowed%s", $section, $suffix));
                    }
                } else {
                    // Entire section disallowed
                    throw new SpecificationException(sprintf("Key %s not allowed%s", $section, $suffix));
                }
            }
        }
    }
}