<?php

declare(strict_types=1);

namespace Atlas\ReportRenderBundle\Tests\DependencyInjection;

use Atlas\ReportRenderBundle\DependencyInjection\ReportRenderExtension;
use Atlas\ReportRenderBundle\Service\ReportRenderer;
use PHPUnit\Framework\TestCase;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Extension\ExtensionInterface;

final class ReportRenderExtensionTest extends TestCase
{
    public function test_load_registers_services(): void
    {
        $container = new ContainerBuilder();

        $extension = new ReportRenderExtension();
        $extension->load([], $container);

        self::assertTrue($container->hasDefinition(ReportRenderer::class));
    }

    public function test_prepend_without_twig_does_nothing(): void
    {
        $container = new ContainerBuilder();

        $extension = new ReportRenderExtension();
        $extension->prepend($container);

        self::assertSame([], $container->getExtensionConfig('twig'));
    }

    public function test_prepend_registers_twig_path(): void
    {
        $container = new ContainerBuilder();

        // Minimal Twig extension stub that satisfies ContainerBuilder
        $twig = new class implements ExtensionInterface {
            public function getAlias(): string { return 'twig'; }
            public function load(array $configs, ContainerBuilder $container): void {}
            public function getNamespace(): string { return 'twig'; }
            public function getXsdValidationBasePath(): bool { return false; }
            public function getXsdValidationFile(): ?string { return null; }
        };

        $container->registerExtension($twig);
        // Mark it as present/loaded so hasExtension('twig') is true
        $container->getExtensions(); // ensure internal init
        $container->loadFromExtension('twig', []); // registers twig as loaded

        $extension = new ReportRenderExtension();
        $extension->prepend($container);

        $configs = $container->getExtensionConfig('twig');

        self::assertNotEmpty($configs);

        // our prepend adds ['paths' => [ <templates-path> => 'ReportRender' ]]
        $found = false;
        foreach ($configs as $config) {
            if (isset($config['paths']) && is_array($config['paths'])) {
                foreach ($config['paths'] as $path => $ns) {
                    if ($ns === 'ReportRender') {
                        $found = true;
                        break 2;
                    }
                }
            }
        }

        self::assertTrue($found, 'Expected twig paths to include namespace "ReportRender".');
    }
}