<?php

namespace Atlas\SecurityManagerBundle\Entity\Role;

use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\AuditBundle\Attribute\AuditActor;
use Atlas\AuditBundle\Attribute\AuditTimestamp;
use Atlas\AuditBundle\Attribute\Enum\AuditActionType;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Validator\Constraints\Length;
use Symfony\Component\Validator\Constraints\NotBlank;


use Atlas\SecurityManagerBundle\Repository\Role\RolePermissionRepository;
#[ORM\Entity(repositoryClass: RolePermissionRepository::class)]
#[ORM\Table(name: 'sys_role_permission')]
#[ORM\Index(columns: ['role_id'])]
#[ORM\Index(columns: ['permission_id'])]
#[ORM\UniqueConstraint(columns: ['role_id', 'permission_id'])]
#[UniqueEntity(fields: ['role', 'permission'])]
class RolePermission
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\ManyToOne(targetEntity: Role::class, inversedBy: 'permissions')]
    #[ORM\JoinColumn(name: 'role_id', referencedColumnName: 'id', nullable: false)]
    private(set) Role $role;

    #[ORM\ManyToOne(targetEntity: Permission::class)]
    #[ORM\JoinColumn(name: 'permission_id', referencedColumnName: 'id', nullable: false)]
    private(set) Permission $permission;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[AuditTimestamp(AuditActionType::INSERT)]
    #[AuditTimestamp(AuditActionType::UPDATE)]
    private(set) DateTimeInterface $added;

    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[AuditActor(AuditActionType::INSERT)]
    #[AuditActor(AuditActionType::UPDATE)]
    private(set) string $added_by;

    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[Length(min: 10, max: 255)]
    private(set) string $reason;

    //not in db
    #[AuditTimestamp(AuditActionType::DELETE)]
    private(set) ?DateTimeInterface $deleted = null;
    #[AuditActor(AuditActionType::DELETE)]
    private(set) ?string $deleted_by = null;

    /**
     * @param Role $role
     * @param Permission $permission
     * @param string $addedBy
     * @param string $reason
     */
    public function __construct(Role $role, Permission $permission, string $addedBy, string $reason)
    {
        if (empty($addedBy = mb_trim($addedBy, encoding: 'UTF-8'))) {
            throw new NotBlankException('Granted by should not be empty');
        }

        if (empty($reason = mb_trim($reason, encoding: 'UTF-8'))) {
            throw new NotBlankException('Reason should not be empty');
        }

        $this->role = $role;
        $this->permission = $permission;
        $this->added_by = $addedBy;
        $this->reason = $reason;
        $this->added = new DateTimeImmutable();
    }

    /**
     * @param string $deletedBy
     * @param string $reason
     * @return void
     */
    public function preDelete(string $deletedBy, string $reason): void
    {
        if (empty($deletedBy = mb_trim($deletedBy, encoding: 'UTF-8'))) {
            throw new NotBlankException('Deleted by should not be empty');
        }

        if (empty($reason = mb_trim($reason, encoding: 'UTF-8'))) {
            throw new NotBlankException('Reason should not be empty');
        }

        $this->deleted_by = $deletedBy;
        $this->reason = $reason;
        $this->deleted = new DateTimeImmutable();
    }
}
