<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Entity\User;

use Atlas\AuditBundle\Attribute\AuditActor;
use Atlas\AuditBundle\Attribute\AuditTimestamp;
use Atlas\AuditBundle\Attribute\Enum\AuditActionType;
use Atlas\AuditBundle\Attribute\NotLogged;
use Atlas\SecurityManagerBundle\Exception\LockException;
use Atlas\SecurityManagerBundle\Exception\NoUpdateRequiredException;
use Atlas\SecurityManagerBundle\Exception\User\InternalUserException;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\SecurityManagerBundle\Exception\Validation\ValidationException;
use Atlas\SecurityManagerBundle\Repository\User\UserRepository;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Validator\Constraints as Assert;

#[ORM\Entity(repositoryClass: UserRepository::class)]
#[ORM\Table(name: 'sys_user_account')]
#[UniqueEntity(fields: ['email'])]
#[UniqueEntity(fields: ['username'])]
class User
{
    #[NotLogged]
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\Column(length: 35)]
    #[Assert\NotBlank]
    #[Assert\Length(min: 2, max: 35)]
    private(set) string $firstname;

    #[ORM\Column(length: 35)]
    #[Assert\NotBlank]
    #[Assert\Length(min: 2, max: 35)]
    private(set) string $lastname;

    #[ORM\Column(length: 255, unique: true)]
    #[Assert\Email]
    #[Assert\NotBlank]
    private(set) string $email;

    #[ORM\Column(length: 10, unique: true)]
    #[Assert\NotBlank]
    #[Assert\Length(max: 10)]
    private(set) string $username;

    #[ORM\Column]
    private(set) bool $internal;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE, nullable: true)]
    private(set) ?DateTimeInterface $validated = null;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE, nullable: true)]
    private(set) ?DateTimeInterface $locked = null;

    #[NotLogged]
    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[AuditTimestamp(AuditActionType::INSERT)]
    private(set) DateTimeInterface $created;

    #[NotLogged]
    #[ORM\Column(length: 255)]
    #[Assert\NotBlank]
    #[AuditActor(AuditActionType::INSERT)]
    private(set) string $created_by;

    #[NotLogged]
    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[AuditTimestamp(AuditActionType::UPDATE)]
    private(set) DateTimeInterface $modified;

    #[NotLogged]
    #[ORM\Column(length: 255)]
    #[Assert\NotBlank]
    #[AuditActor(AuditActionType::UPDATE)]
    private(set) string $modified_by;

    #[NotLogged]
    #[ORM\Column(length: 255)]
    #[Assert\Length(min: 10, max: 255)]
    #[Assert\NotBlank]
    private(set) string $reason;

    #[ORM\OneToMany(targetEntity: UserRole::class, mappedBy: 'user', orphanRemoval: true)]
    private(set) Collection $user_roles;

    //virtual variables
    public string $full_name {
        get => $this->firstname . ' ' . $this->lastname;
    }
    public string $inverted_name {
        get => $this->lastname . ', ' . $this->firstname;
    }
    public string $initials {
        get => $this->firstname[0] . $this->lastname[0];
    }
    public string $initial_lastname {
        get => $this->firstname[0] . ' ' . $this->lastname;
    }
    public bool $is_internal {
        get => $this->internal;
    }
    public bool $is_validated {
        get => isset($this->validated);
    }
    public bool $is_locked {
        get => isset($this->locked);
    }

    /**
     * @param string $username
     * @param string $firstname the users first name
     * @param string $lastname the users last name
     * @param string $email the users email address
     * @param string $actionBy who created the user
     * @param bool $internal
     * @throws NotBlankException
     */
    public function __construct(
        string $username,
        string $firstname,
        string $lastname,
        string $email,
        string $actionBy,
        bool $internal = false,
    ) {
        $timestamp = new DateTimeImmutable();

        if (empty($firstname = mb_trim($firstname, encoding: 'UTF-8'))) {
            throw new NotBlankException('Firstname should not be empty');
        }

        if (empty($lastname = mb_trim($lastname, encoding: 'UTF-8'))) {
            throw new NotBlankException('Lastname should not be empty');
        }

        if (empty($email = mb_trim($email, encoding: 'UTF-8'))) {
            throw new NotBlankException('Email should not be empty');
        }

        if (empty($username = mb_trim($username, encoding: 'UTF-8'))) {
            throw new NotBlankException('Username should not be empty');
        }

        if (empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw new NotBlankException('Action by should not be empty');
        }

        $this->firstname = $firstname;
        $this->lastname = $lastname;
        $this->email = mb_strtolower($email, encoding: 'UTF-8');
        $this->username = mb_strtolower($username, encoding: 'UTF-8');
        $this->internal = $internal;
        if ($internal || $username === 'system') {
            $this->validated = $timestamp;
        }
        $this->user_roles = new ArrayCollection();
        $this->created = $this->modified = $timestamp;
        $this->created_by = $this->modified_by = $actionBy;
        $this->reason = 'Initial creation of user';
    }

    /**
     * @param string $actionBy
     * @param string $reason
     * @param string|null $firstname
     * @param string|null $lastname
     * @param string|null $email
     * @throws NotBlankException
     * @throws NoUpdateRequiredException
     * @throws InternalUserException
     */
    public function update(
        string $actionBy,
        string $reason,
        ?string $firstname = null,
        ?string $lastname = null,
        ?string $email = null
    ): void {
        $timestamp = new DateTimeImmutable();

        if (empty($reason = mb_trim($reason, encoding: 'UTF-8'))) {
            throw new NotBlankException('Reason should not be empty');
        }

        if (empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw new NotBlankException('Action by should not be empty');
        }

        if($this->is_internal) throw new InternalUserException('You cannot edit an internal user');

        $changed = false;

        if (!empty($firstname) && $this->firstname !== $firstname = mb_trim($firstname, encoding: 'UTF-8')) {
            $this->firstname = $firstname;
            $changed = true;
        }

        if (!empty($lastname) && $this->lastname !== $lastname = mb_trim($lastname, encoding: 'UTF-8')) {
            $this->lastname = $lastname;
            $changed = true;
        }

        if (!empty($email) && $this->email !== $email = mb_trim($email, encoding: 'UTF-8')) {
            $this->email = $email;
            $changed = true;
        }

        if ($changed) {
            $this->modified = $timestamp;
            $this->modified_by = $actionBy;
            $this->reason = $reason;
        }

        if(! $changed) throw NoUpdateRequiredException::forId('User', $this->id);
    }

    /**
     * @param string|null $firstname
     * @param string|null $lastname
     * @param string|null $email
     * @return void
     * @throws InternalUserException
     */
    public function ldapSyncUser(
        ?string $firstname = null,
        ?string $lastname = null,
        ?string $email = null
    ): void {
        $timestamp = new DateTimeImmutable();

        if(! $this->is_internal) throw new InternalUserException('You cannot sync an external user');

        $changed = false;

        if (!empty($firstname) && $this->firstname !== $firstname = mb_trim($firstname, encoding: 'UTF-8')) {
            $this->firstname = $firstname;
            $changed = true;
        }

        if (!empty($lastname) && $this->lastname !== $lastname = mb_trim($lastname, encoding: 'UTF-8')) {
            $this->lastname = $lastname;
            $changed = true;
        }

        if (!empty($email) && $this->email !== $email = mb_trim($email, encoding: 'UTF-8')) {
            $this->email = $email;
            $changed = true;
        }

        if ($changed) {
            $this->modified = $timestamp;
            $this->modified_by = 'LDAP';
            $this->reason = 'LDAP user synced';
        }

        //if not changed we just fall through we don't care about syncs
    }

    /**
     * @param string $actionBy
     * @param string $reason
     * @param bool $lock
     * @throws NotBlankException
     * @throws LockException
     */
    public function lock(string $actionBy, string $reason, bool $lock = true): void
    {
        if (empty($reason = mb_trim($reason, encoding: 'UTF-8'))) {
            throw new NotBlankException('Reason should not be empty');
        }
        if (empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw new NotBlankException('Action by should not be empty');
        }

        $timestamp = new DateTimeImmutable();

        if ($this->is_locked !== $lock) {
            $this->locked = $lock ? $timestamp : null;
            $this->modified = $timestamp;
            $this->modified_by = $actionBy;
            $this->reason = $reason;
        }
        else {
            throw LockException::forResource('User', $lock);
        }
    }

    /**
     * @throws ValidationException
     */
    public function validate(): void
    {
        if($this->is_validated) throw new ValidationException('User already validated');

        $this->modified = $this->validated = new DateTimeImmutable();
        $this->modified_by = 'System';
        $this->reason = 'User automatically validated';

    }
}
