<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Listener;

use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\EventDispatcher\Attribute\AsEventListener;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\Routing\RouterInterface;

#[AsEventListener(event: 'kernel.request', priority: -10)]
final class SlidingSessionListener
{
    private const int IDLE_TIMEOUT = 20 * 60;          // 20 minutes
    private const int ABSOLUTE_TIMEOUT = 4 * 60 * 60;  // 4 hours
    private const int MIGRATE_EVERY = 5 * 60;          // rotate every 5 mins
    private const int PERM_REFRESH_EVERY = 15 * 60;    // refresh perms every 15 mins

    public function __construct(
        private readonly Security $security,
        private readonly RouterInterface $router,
    ) {}

    public function __invoke(RequestEvent $event): void
    {
        $request = $event->getRequest();

        $path = $request->getPathInfo();

        // skip obvious loops / internals
        if ($path === '/login' || $path === '/logout' || str_starts_with($path, '/_')) {
            return;
        }

        // must be logged in
        if ($this->security->getUser() === null) {
            return;
        }

        $session = $request->getSession();
        $now = time();

        // init timestamps once
        $loginAt = (int) $session->get('auth.login_at', 0);
        if ($loginAt === 0) {
            $loginAt = $now;
            $session->set('auth.login_at', $loginAt);
        }

        $lastSeen = (int) $session->get('auth.last_seen', 0);
        if ($lastSeen === 0) {
            $lastSeen = $now;
            $session->set('auth.last_seen', $lastSeen);
        }

        // compute BEFORE mutation
        $idle = $now - $lastSeen;
        $age  = $now - $loginAt;

        // enforce timeouts BEFORE sliding
        if ($idle > self::IDLE_TIMEOUT || $age > self::ABSOLUTE_TIMEOUT) {
            $session->invalidate();
            $event->setResponse(new RedirectResponse($this->router->generate('_logout_main')));
            return;
        }

        if ($request->isXmlHttpRequest()
            || $request->headers->has('Turbo-Frame')
            || $request->headers->get('Accept') === 'text/vnd.turbo-stream.html'
        ) {
            return;
        }

        // slide on ANY authenticated request (main or subrequest)
        $session->set('auth.last_seen', $now);

        // perm refresh + migrate only on MAIN request (avoid subrequest spam)
        if ($event->isMainRequest()) {

            $nextPerm = (int) $session->get('auth.next_perm_refresh', 0);

            // if nextPerm is absurdly far away (e.g. from previous config), reschedule
            if ($nextPerm > ($now + self::PERM_REFRESH_EVERY * 2)) {
                $nextPerm = 0;
                $session->remove('auth.next_perm_refresh');
            }

            if ($nextPerm === 0) {
                $session->set('auth.next_perm_refresh', $now + self::PERM_REFRESH_EVERY);
            } elseif ($now >= $nextPerm) {
                $session->set('auth.force_perm_refresh', 1);
                $session->set('auth.next_perm_refresh', $now + self::PERM_REFRESH_EVERY);
            }

            $lastMig = (int) $session->get('auth.last_migration', 0);
            if (($now - $lastMig) > self::MIGRATE_EVERY) {
                $session->migrate(true);
                $session->set('auth.last_migration', $now);
            }
        }
    }
}
