<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Repository\Role;

use Atlas\SecurityManagerBundle\Contract\FormInterface;
use Atlas\SecurityManagerBundle\Entity\Role\RoleForm;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;
use Psr\Cache\InvalidArgumentException;
use Symfony\Contracts\Cache\CacheInterface;
use Symfony\Contracts\Cache\ItemInterface;

/**
 * @extends ServiceEntityRepository<RoleForm>
 */
class RoleFormRepository extends ServiceEntityRepository
{
    private const string CACHE_KEY = 'security_form_permissions_map';
    private const int TTL = 86400;

    private CacheInterface $cache;

    public function __construct(
        ManagerRegistry $registry,
        CacheInterface $cache
    ) {
        $this->cache = $cache;
        parent::__construct($registry, RoleForm::class);
    }

    /**
     * @param bool $forceRebuild
     * @return array
     * @throws InvalidArgumentException
     */
    public function getPermissionMap(bool $forceRebuild = false): array
    {
        if ($forceRebuild) {
            $this->clearPermissionMapCache();
        }

        return $this->cache->get(self::CACHE_KEY, function (ItemInterface $item): array {
            $item->expiresAfter(self::TTL);

            $forms = $this->createQueryBuilder('rf')
                ->select('r.code AS role_code', 'f.code AS form_code')
                ->join('rf.role', 'r')
                ->join('rf.form', 'f')
                ->where('r.active = true')
                ->andWhere('f.active = true')
                ->orderBy('r.code', 'ASC')
                ->addOrderBy('f.code', 'ASC')
                ->getQuery()
                ->getScalarResult();

            $map = [];

            foreach ($forms as $row) {
                $map[$row['role_code']][] = $row['form_code'];
            }

            return $map;
        });
    }

    /**
     * @param int $roleId
     * @return array
     */
    public function findFormsForRoleId(int $roleId): array
    {
        return $this->getEntityManager()
            ->createQueryBuilder()
            ->select('f')
            ->from(FormInterface::class, 'f')
            ->innerJoin(RoleForm::class, 'rf', 'WITH', 'rf.form = f')
            ->where('IDENTITY(rf.role) = :rid')
            ->setParameter('rid', $roleId)
            ->orderBy('f.name', 'ASC')
            ->addOrderBy('f.code', 'ASC')
            ->getQuery()
            ->getResult();
    }

    /**
     * Manually clear the form permission map cache.
     *
     * @throws InvalidArgumentException
     */
    public function clearPermissionMapCache(): void
    {
        $this->cache->delete(self::CACHE_KEY);
    }
}
