<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Repository\Role;

use Atlas\SecurityManagerBundle\Entity\Role\Role;
use Atlas\SecurityManagerBundle\Exception\Role\RoleNotFoundException;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\QueryBuilder;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @extends ServiceEntityRepository<Role>
 */
class RoleRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Role::class);
    }

    /**
     * @param int $id
     * @return Role
     * @throws RoleNotFoundException
     */
    public function findOrThrow(int $id): Role
    {
        $role = $this->find($id);
        if ($role === null) {
            throw RoleNotFoundException::fromId($id);
        }
        return $role;
    }

    public function findAllQueryBuilder(
        ?string $filter = null,
        ?string $sort = null,
        ?string $direction = null,
        ?string $search = null
    ): QueryBuilder {
        $query = $this->createQueryBuilder('r');

        // Filter
        if (!empty($filter)) {
            switch ($filter) {
                case 'active':
                    $query->andWhere('r.active = true');
                    break;
                case 'inactive':
                    $query->andWhere('r.active = false');
                    break;
            }
        }

        // Search (case-insensitive, UTF-8 safe)
        if (!empty($search)) {
            $search = mb_trim((string)$search, encoding: 'UTF-8');
            if ($search !== '') {
                $needle = '%' . mb_strtolower($search, encoding: 'UTF-8') . '%';
                $query->andWhere(
                    $query->expr()->orX(
                        $query->expr()->like('LOWER(r.name)', ':search'),
                        $query->expr()->like('LOWER(r.code)', ':search')
                    )
                )->setParameter('search', $needle);
            }
        }

        // Sort + direction (validated)
        $allowedSorts = [
            'name' => 'r.name',
            'code' => 'r.code',
            'id' => 'r.id'
        ];

        if (!empty($sort)) {
            $sort = mb_strtolower($sort, encoding: 'UTF-8');
        }

        if (!array_key_exists($sort, $allowedSorts)) {
            $sort = 'id';
        }

        $direction = (mb_strtoupper((string) $direction, encoding: 'UTF-8') === 'DESC') ? 'DESC' : 'ASC';

        $orderField = $allowedSorts[$sort] ?? $allowedSorts['id'];

        $query->orderBy($orderField, $direction);

        return $query;
    }
}
