<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Repository\User;

use Atlas\SecurityManagerBundle\Entity\User\User;
use Atlas\SecurityManagerBundle\Exception\User\UserNotFoundException;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\DBAL\ArrayParameterType;
use Doctrine\ORM\QueryBuilder;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @extends ServiceEntityRepository<User>
 */
final class UserRepository extends ServiceEntityRepository
{
    /**
     * @param ManagerRegistry $registry
     */
    public function __construct(
        ManagerRegistry $registry
    )
    {
        parent::__construct($registry, User::class);
    }

    /**
     * @param int $id
     * @return User
     * @throws UserNotFoundException
     */
    public function findOrThrow(int $id): User
    {
        $participant = $this->find($id);
        if ($participant === null) {
            throw UserNotFoundException::fromId($id);
        }
        return $participant;
    }

    /**
     * @param string $identifier
     * @return User|null
     */
    public function findOneByUsernameOrEmail(string $identifier): ?User
    {
        $id = mb_strtolower (mb_trim($identifier, encoding: 'UTF-8'), encoding: 'UTF-8');

        return $this->createQueryBuilder('u')
            ->andWhere('LOWER(u.email) = :id OR LOWER(u.username) = :id')
            ->setParameter('id', $id)
            // deterministic tie-break if duplicates exist
            ->addOrderBy('u.validated', 'DESC')
            ->addOrderBy('u.id', 'DESC')
            ->setMaxResults(1)
            ->getQuery()
            ->getOneOrNullResult();
    }

    /**
     * @param string $identifier
     * @return array
     */
    public function findAllByUsernameOrEmail(string $identifier): array
    {
        $id = mb_strtolower (mb_trim($identifier, encoding: 'UTF-8'), encoding: 'UTF-8');

        return $this->createQueryBuilder('u')
            ->andWhere('LOWER(u.email) = :id OR LOWER(u.username) = :id')
            ->setParameter('id', $id)
            ->getQuery()
            ->getResult();
    }


    /**
     * @param string $username
     * @return User|null
     */
    public function findOneByUsername(string $username): ?User
    {
        $uname = mb_strtolower (mb_trim($username, encoding: 'UTF-8'), encoding: 'UTF-8');

        return $this->createQueryBuilder('u')
            ->andWhere('LOWER(u.username) = :uname')
            ->setParameter('uname', $uname)
            ->setMaxResults(1)
            ->getQuery()
            ->getOneOrNullResult();
    }

    /**
     * @param string|null $filter
     * @param string|null $sort
     * @param string|null $direction
     * @param string|null $search
     * @param int[]|null $locationIds
     * @param array $excludeIds
     * @return QueryBuilder
     */
    public function findAllQueryBuilder(
        ?string $filter = null,
        ?string $sort = null,
        ?string $direction = null,
        ?string $search = null,
        ?array $locationIds = null,
        array $excludeIds = []
    ): QueryBuilder {

        $query = $this->createQueryBuilder('u');

        //restrict by location
        if ($locationIds !== null) {
            $locationIds = array_values(array_unique(array_map('intval', $locationIds)));
            if ($locationIds === []) {
                $query->andWhere('1 = 0');
                return $query;
            } else {
                $query->innerJoin('u.user_roles', 'ur')   // OneToMany<UserRole> on User
                    ->innerJoin('ur.location', 'location')      // ManyToOne<Location> on UserRole
                    ->andWhere($query->expr()->in('location.id', ':location_ids'))
                    ->setParameter('location_ids', $locationIds, ArrayParameterType::INTEGER)
                    ->distinct();
            }
        }

        // Exclude IDs
        if (!empty($excludeIds)) {
            $excludeIds = array_values(array_unique(array_map('intval', $excludeIds)));
            $query->andWhere($query->expr()->notIn('u.id', ':exclude'))
                ->setParameter('exclude', $excludeIds, ArrayParameterType::INTEGER);
        }

        $direction = (mb_strtoupper((string) $direction, encoding: 'UTF-8') === 'DESC') ? 'DESC' : 'ASC';

        // Sort + direction (validated)
        $allowedSorts = [
            'firstname' => 'u.firstname',
            'lastname' => 'u.lastname',
            'email' => 'u.email',
            'id' => 'u.id'
        ];

        if (!empty($sort)) {
            $sort = mb_strtolower($sort, encoding: 'UTF-8');
        }

        if (!array_key_exists($sort, $allowedSorts)) {
            $sort = 'id';
        }

        $orderField = $allowedSorts[$sort] ?? $allowedSorts['id'];

        $query->orderBy($orderField, $direction);

        if (!empty($search)) {
            $search = mb_trim((string)$search, encoding: 'UTF-8');
            if ($search !== '') {
                $needle = '%' . mb_strtolower($search, encoding: 'UTF-8') . '%';
                // use andWhere to avoid unintentionally overwriting earlier where() calls
                $query->andWhere(
                    $query->expr()->orX(
                        $query->expr()->like('LOWER(u.username)', ':search'),
                        $query->expr()->like('LOWER(u.firstname)', ':search'),
                        $query->expr()->like('LOWER(u.lastname)', ':search'),
                        $query->expr()->like('LOWER(u.email)', ':search')
                    )
                )->setParameter('search', $needle);
            }
        }

        $filter = !empty($filter) ? mb_strtolower($filter, encoding: 'UTF-8') : null;

        switch ($filter) {
            case 'active':
                $query->andWhere(
                    $query->expr()->andX(
                        $query->expr()->isNull('u.locked'),
                        $query->expr()->isNotNull('u.validated')
                    )
                );
                break;
            case 'locked':
                $query->andWhere($query->expr()->isNotNull('u.locked'));
                break;
            case 'unvalidated':
                $query->andWhere($query->expr()->isNull('u.validated'));
                break;
        }

        return $query;
    }

    /**
     * @param string $prefix
     * @return int
     */
    public function countUsernamesStartingWith(string $prefix): int
    {
        return (int)$this->createQueryBuilder('u')
            ->select('COUNT(u.id)')
            ->where('LOWER(u.username) LIKE :prefix')
            ->setParameter('prefix', mb_strtolower($prefix, encoding: 'UTF-8') . '%')
            ->getQuery()
            ->getSingleScalarResult();
    }

}
