<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Controller;

use Atlas\SecurityManagerBundle\Controller\Concern\UrlParamsTrait;
use Atlas\SecurityManagerBundle\Entity\Role\Role;
use Atlas\SecurityManagerBundle\Service\Role\RoleManager;
use Modules\Paginator\Service\PaginatorLinks;
use Symfony\Bridge\Doctrine\Attribute\MapEntity;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\ExpressionLanguage\Expression;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;


#[IsGranted('ROLE_USER')]
final class RoleController extends AbstractController
{
    use UrlParamsTrait;

    public function __construct(private readonly RoleManager $manager) {}

    /**
     * @param Request $request
     * @param PaginatorLinks $paginationLinks
     * @return Response
     */
    #[Route('/role', name: 'security_role', methods: ['GET'])]
    #[IsGranted(
        attribute: 'permission',
        subject: [
            'permission' => new Expression('"role"')
        ]
    )]
    public function index(
        Request $request,
        PaginatorLinks $paginationLinks
    ): Response
    {
        $queryString = $this->getRoleQueryString($request);

        $paginator = $this->manager->getRolesPage($queryString);

        // defensive defaults in case paginator shape changes
        $items = $paginator['items'] ?? [];
        $pages = $paginator['pages'] ?? 1;
        $page = $paginator['page'] ?? ($queryString['p'] ?? 1);

        //only use this at the end so queryString is cleaned
        $queryString = $this->urlParams($queryString);

        $links = $paginationLinks->generateLinks($pages, (int)$page, $this->generateUrl('security_role', $queryString));

        return $this->render('@SecurityManager/role/index.html.twig', [
            'roles' => $items,
            'pagination_links' => $links,
            'query_string' => $queryString
        ]);
    }

    /**
     * @param Request $request
     * @return Response
     */
    #[Route('/role/add', name: 'security_role_add', methods: ['GET'])]
    #[IsGranted(
        attribute: 'permission',
        subject: [
            'permission' => new Expression('"role.add"')
        ]
    )]
    public function add(Request $request): Response
    {
        $queryString = $this->getRoleQueryString($request);

        $queryString = $this->urlParams($queryString);

        return $this->render('@SecurityManager/role/add.html.twig', [
            'query_string' => $queryString
        ]);
    }

    /**
     * @param Request $request
     * @param Role $role
     * @return Response
     */
    #[Route('/role/edit/{id<\d+>}', name: 'security_role_edit', methods: ['GET'])]
    #[IsGranted(
        attribute: 'permission',
        subject: [
            'permission' => new Expression('"role.edit"'),
            'role' => new Expression('args["role"].code')
        ]
    )]
    public function edit(
        Request $request,
        #[MapEntity(mapping: ['id' => 'id'])] Role $role
    ): Response
    {
        $queryString = $this->getRoleQueryString($request);

        $queryString = $this->urlParams($queryString);

        return $this->render('@SecurityManager/role/edit.html.twig', [
            'role' => $role,
            'query_string' => $queryString
        ]);
    }

    /**
     * @param Role $role
     * @param Request $request
     * @return Response
     */
    #[Route(path: '/role/{id<\d+>}', name: 'security_role_view', methods: ['GET'])]
    #[IsGranted(
        attribute: 'permission',
        subject: [
            'permission' => new Expression('"role"')
        ]
    )]
    public function view(
        Request $request,
        #[MapEntity(mapping: ['id' => 'id'])] Role $role): Response
    {
        $queryString = $this->getRoleQueryString($request);

        $queryString = $this->urlParams($queryString);

        return $this->render('@SecurityManager/role/view.html.twig', [
            'role' => $role,
            'query_string' => $queryString,
            'action' => 'view',
        ]);
    }

    private function getRoleQueryString(Request $request): array
    {
        //if we have role specific we'd put it here
        //can pass as array optionally here
        return $this->getQueryString($request);
    }
}
