<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Controller;

use Atlas\SecurityManagerBundle\Controller\Concern\UrlParamsTrait;
use Atlas\SecurityManagerBundle\Dto\User\LdapUserSearchDto;
use Atlas\SecurityManagerBundle\Entity\User\User;
use Atlas\SecurityManagerBundle\Exception\User\InternalUserException;
use Atlas\SecurityManagerBundle\Exception\User\UserNotFoundException;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\SecurityManagerBundle\Form\User\InternalUserForm;
use Atlas\SecurityManagerBundle\Security\User as SecurityUser;
use Atlas\SecurityManagerBundle\Service\User\LdapSearch;
use Atlas\SecurityManagerBundle\Service\User\UserManager;
use LogicException;
use Modules\Paginator\Service\PaginatorLinks;
use Symfony\Bridge\Doctrine\Attribute\MapEntity;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\DependencyInjection\Exception\ServiceNotFoundException;
use Symfony\Component\ExpressionLanguage\Expression;
use Symfony\Component\Form\Exception\OutOfBoundsException;
use Symfony\Component\Form\Exception\RuntimeException;
use Symfony\Component\Form\FormError;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Ldap\Exception\ConnectionException;
use Symfony\Component\Ldap\Exception\LdapException;
use Symfony\Component\Ldap\Exception\NotBoundException;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Core\Exception\AccessDeniedException;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\Component\Validator\Exception\InvalidArgumentException;

#[IsGranted('ROLE_USER')]
final class UserController extends AbstractController
{
    use UrlParamsTrait;

    /**
     * @param LdapSearch $ldap_search
     * @param UserManager $user_manager
     * @param bool $useLdap
     */
    public function __construct(
        private readonly LdapSearch $ldap_search,
        private readonly UserManager $user_manager,
        #[Autowire(param: 'security.ldap.enabled')] private readonly bool $useLdap = false,
    )
    {
    }

    /**
     * @param Request $request
     * @param PaginatorLinks $paginationLinks
     * @param UserManager $manager
     * @return Response
     * @throws LogicException
     * @throws ServiceNotFoundException
     */
    #[Route('/user', name: 'security_user', methods: ['GET'])]
    #[IsGranted(
        attribute: 'permission',
        subject: [
            'permission' => new Expression('"usr"'),
        ]
    )]
    public function index(
        Request $request,
        PaginatorLinks $paginationLinks,
        UserManager $manager
    ): Response
    {

        $queryString = $this->getUserQueryString($request);

        $paginator = $manager->getUsersPage($queryString);

        // defensive defaults in case paginator shape changes
        $items = $paginator['items'] ?? [];
        $pages = $paginator['pages'] ?? 1;
        $page = $paginator['page'] ?? ($queryString['p'] ?? 1);

        //only use this at the end so queryString is cleaned
        $queryString = $this->urlParams($queryString);

        $links = $paginationLinks->generateLinks($pages, (int)$page, $this->generateUrl('security_user', $queryString));

        //check to see if they can self-manage
        $selfManage = $this->isGranted('permission', ['permission' => 'usr.self']);

        return $this->render('@SecurityManager/user/index.html.twig', [
            'users' => $items,
            'pagination_links' => $links,
            'query_string' => $queryString,
            'self_manage' => $selfManage,
            'use_ldap' => $this->useLdap,
        ]);
    }

    /**
     * @param Request $request
     * @return Response
     */
    #[Route('/user/external', name: 'security_user_add_external', methods: ['GET'])]
    #[IsGranted(
        attribute: 'permission',
        subject: [
            'permission' => new Expression('"usr.add"'),
        ]
    )]
    public function addExternal(Request $request): Response
    {

        $queryString = $this->getUserQueryString($request);
        $queryString = $this->urlParams($queryString);

        return $this->render('@SecurityManager/user/add_external.html.twig', [
            'query_string' => $queryString,
            'use_ldap' => $this->useLdap,
        ]);
    }

    /**
     * @param Request $request
     * @return Response
     * @throws LdapException
     * @throws OutOfBoundsException
     * @throws LogicException
     * @throws RuntimeException
     */
    #[Route('/user/internal', name: 'security_user_add_internal', methods: ['GET', 'POST'])]
    #[IsGranted(
        attribute: 'permission',
        subject: [
            'permission' => new Expression('"usr.add"'),
        ]
    )]
    public function addInternal(Request $request): Response
    {
        if (!$this->useLdap) throw new LdapException('LDAP must be enabled to add internal users');

        $queryString = $this->getUserQueryString($request);
        $queryString = $this->urlParams($queryString);

        $form = $this->createForm(InternalUserForm::class);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {

            /** @var LdapUserSearchDto $dto */
            $dto = $form->getData();

            try {

                $securityUser = $this->getUser();

                if (!$securityUser instanceof SecurityUser) {
                    throw new AccessDeniedException('You must be logged in to add an internal user');
                }

                $user = $this->ldap_search->search($dto->identifier);
                $user = $this->user_manager->createInternalUser($user, $securityUser->email);

                return $this->redirectToRoute('security_user_view', ['id' => $user->id]);

            } catch (InternalUserException|UserNotFoundException|NotBlankException $e) {
                $form->get('identifier')->addError(new FormError($e->getMessage()));
            } catch (ConnectionException|NotBoundException) {
                $form->get('identifier')->addError(new FormError('Error connecting to the LDAP server, please try again later'));
            } catch (LdapException) {
                $form->get('identifier')->addError(new FormError('LDAP look up failed'));
            } catch (InvalidArgumentException|LogicException) {
                $form->get('identifier')->addError(new FormError('Validator could not be initialised'));
            }
        }

        return $this->render('@SecurityManager/user/add_internal.html.twig', [
            'form' => $form->createView(),
            'query_string' => $queryString,
            'use_ldap' => $this->useLdap,
        ]);
    }

    /**
     * @param Request $request
     * @param User $user
     * @return Response
     */
    #[Route('/user/edit/{id<\d+>}', name: 'security_user_edit', methods: ['GET'])]
    #[IsGranted(
        attribute: 'permission',
        subject: [
            'permission' => new Expression('"usr.edit"'),
            'user' => new Expression('args["user"].id')
        ]
    )]
    public function edit(
        Request $request,
        #[MapEntity(mapping: ['id' => 'id'])] User $user
    ): Response
    {

        $queryString = $this->getUserQueryString($request);
        $queryString = $this->urlParams($queryString);

        return $this->render('@SecurityManager/user/edit_external.html.twig', [
            'user' => $user,
            'query_string' => $queryString,
            'use_ldap' => $this->useLdap
        ]);
    }

    /**
     * @param Request $request
     * @param User $user
     * @return Response
     * @throws LogicException
     */
    #[Route('/user/{id<\d+>}', name: 'security_user_view', methods: ['GET'])]
    #[IsGranted(
        attribute: 'permission',
        subject: [
            'permission' => new Expression('"usr"'),
        ]
    )]
    public function view(
        Request $request,
        #[MapEntity(mapping: ['id' => 'id'])] User $user
    ): Response
    {
        $queryString = $this->getUserQueryString($request);
        $queryString = $this->urlParams($queryString);

        //check to see if they can self-manage
        $selfManage = $this->isGranted('permission', ['permission' => 'usr.self']);

        return $this->render('@SecurityManager/user/view.html.twig', [
            'user' => $user,
            'query_string' => $queryString,
            'self_manage' => $selfManage,
            'use_ldap' => $this->useLdap,
        ]);
    }

    private function getUserQueryString(Request $request): array
    {
        //if we have user specific we'd put it here,
        //can pass as array optionally here
        return $this->getQueryString($request);
    }
}
