<?php

namespace Atlas\SecurityManagerBundle\Entity\User;

use Atlas\SecurityManagerBundle\Contract\LocationInterface;
use Atlas\SecurityManagerBundle\Entity\Role\Role;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\SecurityManagerBundle\Repository\User\UserRoleRepository;
use Atlas\AuditBundle\Attribute\AuditActor;
use Atlas\AuditBundle\Attribute\AuditTimestamp;
use Atlas\AuditBundle\Attribute\Enum\AuditActionType;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Validator\Constraints\Length;
use Symfony\Component\Validator\Constraints\NotBlank;


#[ORM\Entity(repositoryClass: UserRoleRepository::class)]
#[ORM\Table(name: 'sys_user_role')]
#[ORM\Index(columns: ['user_id'])]
#[ORM\Index(columns: ['role_id'])]
#[ORM\Index(columns: ['location_id'])]
#[ORM\UniqueConstraint(columns: ['user_id', 'role_id', 'location_id'])]
#[UniqueEntity(fields: ['user', 'role', 'location'])]
class UserRole
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\ManyToOne(targetEntity: User::class, inversedBy: 'user_roles')]
    #[ORM\JoinColumn(name: 'user_id', referencedColumnName: 'id', nullable: false)]
    private(set) User $user;

    #[ORM\ManyToOne(Role::class)]
    #[ORM\JoinColumn(name: 'role_id', referencedColumnName: 'id', nullable: false)]
    private(set) Role $role;

    #[ORM\ManyToOne(targetEntity: LocationInterface::class)]
    #[ORM\JoinColumn(name: 'location_id', referencedColumnName: 'id', nullable: true)]
    private(set) ?LocationInterface $location = null;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[AuditTimestamp(AuditActionType::INSERT)]
    #[AuditTimestamp(AuditActionType::UPDATE)]
    private(set) DateTimeInterface $granted;

    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[AuditActor(AuditActionType::INSERT)]
    #[AuditActor(AuditActionType::UPDATE)]
    private(set) string $granted_by;

    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[Length(min: 10, max: 255)]
    private(set) string $reason;

    //not in db
    #[AuditActor(AuditActionType::DELETE)]
    private(set) ?string $deleted_by = null;
    #[AuditTimestamp(AuditActionType::DELETE)]
    private(set) ?DateTimeInterface $deleted = null;

    /**
     * @param User $user
     * @param Role $role
     * @param string $granted_by
     * @param string $reason
     * @param LocationInterface|null $location
     * @throws NotBlankException
     */
    public function __construct(User $user, Role $role, string $granted_by, string $reason, ?LocationInterface $location = null)
    {
        if (empty($granted_by = mb_trim($granted_by, encoding: 'UTF-8'))) {
            throw new NotBlankException('Granted by should not be empty');
        }

        if (empty($reason = mb_trim($reason, encoding: 'UTF-8'))) {
            throw new NotBlankException('Reason should not be empty');
        }

        $this->user = $user;
        $this->role = $role;
        $this->location = $location;
        $this->granted = new DateTimeImmutable();
        $this->granted_by = $granted_by;
        $this->reason = $reason;
    }

    /**
     * @param string $actionBy
     * @param string $reason
     *
     * @return void
     * @throws NotBlankException
     */
    public function preDelete(string $actionBy, string $reason): void
    {
        if (empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw new NotBlankException('Actioned by should not be empty');
        }

        if (empty($reason = mb_trim($reason, encoding: 'UTF-8'))) {
            throw new NotBlankException('Reason should not be empty');
        }

        $this->deleted_by = $actionBy;
        $this->deleted = new DateTimeImmutable();
        $this->reason = $reason;
    }
}
