<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Repository\User;

use Atlas\SecurityManagerBundle\Contract\LocationInterface;
use Atlas\SecurityManagerBundle\Domain\User\AccessibleLocations;
use Atlas\SecurityManagerBundle\Entity\Role\RolePermission;
use Atlas\SecurityManagerBundle\Entity\User\UserRole;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @extends ServiceEntityRepository<UserRole>
 */
class UserRoleRepository extends ServiceEntityRepository
{

    public const int ALL = 0;
    public const int JUST_LOCATIONS = 1;
    public const int ALL_SITES = 2;

    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, UserRole::class);
    }

    /** @return UserRole[] */
    public function findForUserOrdered(int $userId, ?int $roleId = null, int $find = self::ALL): array
    {
        $qb = $this->createQueryBuilder('ur')
            ->leftJoin('ur.role', 'r')->addSelect('r')
            ->leftJoin('ur.location', 'l')->addSelect('l')
            ->andWhere('ur.user = :uid')
            ->setParameter('uid', $userId)
            ->addOrderBy('r.name', 'ASC')
            ->addOrderBy('l.name', 'ASC');

        if($roleId !== null){
            $qb->andWhere('ur.role = :rid')
                ->setParameter('rid', $roleId);
        }

        if($find === self::JUST_LOCATIONS){
            $qb->andWhere('ur.location IS NOT NULL');
        }
        if($find === self::ALL_SITES){
            $qb->andWhere('ur.location IS NULL');
        }

        return $qb->getQuery()
            ->getResult();
    }

    /**
     * @param int $userId
     * @param int $roleId
     * @param int|null $locationId
     * @return bool
     * @throws NonUniqueResultException
     */
    public function existsFor(int $userId, int $roleId, ?int $locationId = null): bool
    {
        $qb = $this->createQueryBuilder('ur')
            ->select('1')
            ->andWhere('ur.user = :uid')
            ->andWhere('ur.role = :rid')
            ->setParameter('uid', $userId)
            ->setParameter('rid', $roleId)
            ->setMaxResults(1);

        if ($locationId === null) {
            $qb->andWhere('ur.location IS NULL');
        } else {
            $qb->andWhere('ur.location = :lid')->setParameter('lid', $locationId);
        }

        return (bool) $qb->getQuery()->getOneOrNullResult();
    }


    /**
     * @param int $userId
     * @return array<string, string[]> Roles grouped by location (location ID or 'all')
     */
    public function findUserPermissions(int $userId): array
    {
        $rows = $this->createQueryBuilder('ur')
            ->select([
                's.id AS location_id',    // integer or NULL
                'r.id AS role_id',
                'r.code AS role_code',
            ])
            ->join('ur.role', 'r')
            ->leftJoin('ur.location', 's')
            ->where('ur.user = :userId')
            ->andWhere('r.active = true')
            ->setParameter('userId', $userId)
            ->orderBy('location_id', 'ASC')
            ->addOrderBy('role_id',    'ASC')
            ->getQuery()
            ->getScalarResult();

        $roles = [];

        foreach ($rows as $row) {
            $locationCode = $row['location_id'] ?? 'all'; //normalise null location to 'all'
            $roles[$locationCode][] = $row['role_code'];
        }

        return $roles;
    }

    /**
     * Direct assignments only (no children).
     * Returns AccessibleLocations where:
     *  - $vo->is_all === true       => user has NULL-location assignment
     *  - $vo->items is int[]|Location[] => distinct direct locations, depending on $asEntities
     */
    public function findDirectlyAssignedLocations(
        int $userId,
        ?string $permissionCode = null,
        bool $asEntities = false
    ): AccessibleLocations
    {
        // ALL (NULL location) check
        $allQb = $this->createQueryBuilder('ur')
            ->select('COUNT(ur.id)')
            ->where('IDENTITY(ur.user) = :uid')
            ->andWhere('ur.location IS NULL')
            ->setParameter('uid', $userId);

        if ($permissionCode !== null) {
            $allQb->join('ur.role', 'r')
                ->join(RolePermission::class, 'rp', 'WITH', 'rp.role = r')
                ->join('rp.permission', 'p')
                ->andWhere('p.code = :perm')
                ->setParameter('perm', $permissionCode);
        }

        $isAll = (int) $allQb->getQuery()->getSingleScalarResult() > 0;
        if ($isAll) {
            return new AccessibleLocations(is_all: true, items: []);
        }

        if ($asEntities) {
            $query = $this->createQueryBuilder('ur')
                ->select('DISTINCT s')
                ->join('ur.location', 's')
                ->where('IDENTITY(ur.user) = :uid')
                ->setParameter('uid', $userId);

            if ($permissionCode !== null) {
                $query->join('ur.role', 'r')
                    ->join(RolePermission::class, 'rp', 'WITH', 'rp.role = r')
                    ->join('rp.permission', 'p')
                    ->andWhere('p.code = :perm')
                    ->setParameter('perm', $permissionCode);
            }

            /** @var LocationInterface[] $locations */
            $locations = $query->getQuery()->getResult();
            return new AccessibleLocations(is_all: false, items: $locations);
        }

        // IDs
        $query = $this->createQueryBuilder('ur')
            ->select('DISTINCT IDENTITY(ur.location) AS location_id')
            ->where('IDENTITY(ur.user) = :uid')
            ->andWhere('ur.location IS NOT NULL')
            ->setParameter('uid', $userId);

        if ($permissionCode !== null) {
            $query->join('ur.role', 'r')
                ->join(RolePermission::class, 'rp', 'WITH', 'rp.role = r')
                ->join('rp.permission', 'p')
                ->andWhere('p.code = :perm')
                ->setParameter('perm', $permissionCode);
        }

        $rows = $query->getQuery()->getScalarResult();
        $ids  = array_values(array_unique(array_map(fn(array $r): int => (int) $r['location_id'], $rows)));

        return new AccessibleLocations(is_all: false, items: $ids);
    }
}
