<?php
declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Twig\Components\Role;

use Atlas\SecurityManagerBundle\Contract\FormInterface;
use Atlas\SecurityManagerBundle\Entity\Role\Role;
use Atlas\SecurityManagerBundle\Entity\Role\RoleForm;
use Atlas\SecurityManagerBundle\Entity\Role\RolePermission;
use Atlas\SecurityManagerBundle\Exception\ActivateException;
use Atlas\SecurityManagerBundle\Exception\Role\RoleNotFoundException;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\SecurityManagerBundle\Service\Role\RoleManager;
use Atlas\SecurityManagerBundle\Twig\Components\Concern\FlashHelperTrait;
use Atlas\SecurityManagerBundle\Twig\Components\Concern\SecurityUserHelperTrait;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Security\Core\Exception\AccessDeniedException;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveAction;
use Symfony\UX\LiveComponent\Attribute\LiveArg;
use Symfony\UX\LiveComponent\Attribute\LiveProp;
use Symfony\UX\LiveComponent\ComponentToolsTrait;
use Symfony\UX\LiveComponent\DefaultActionTrait;

#[AsLiveComponent(
    name: 'role_details',
    template: '@SecurityManager/components/role/details.html.twig'
)]
final class RoleDetailsComponent
{
    use DefaultActionTrait;
    use ComponentToolsTrait;
    use FlashHelperTrait;
    use SecurityUserHelperTrait;

    #[LiveProp]
    public Role $role;

    #[LiveProp]
    public array $query_string = [];

    #[LiveProp]
    public string $variant = 'view';

    public bool $is_active { get => $this->role->is_active; }

    public function __construct(
        private readonly RoleManager $manager,
        private readonly Security $security,
        private readonly RequestStack $requests
    ) {}

    /**
     * Permissions sorted by permission.code (ASC).
     *
     * @return array<int, RolePermission>
     */
    public array $sorted_permissions {
        get {
            $items = $this->role->permissions->toArray();

            usort(
                $items,
                static fn ($a, $b): int => $a->permission->code <=> $b->permission->code
            );

            return $items;
        }
    }

    /**
     * Forms sorted by form.name (ASC).
     *
     * @return RoleForm[]
     */
    public array $sorted_forms {
        get {
            $items = $this->role->forms->toArray();

            usort(
                $items,
                static fn ($a, $b): int => $a->form->code <=> $b->form->code
            );

            return $items;
        }
    }

    /**
     * Toggle activate/deactivate with a human reason (enforced by the modal)
     */
    #[LiveAction]
    public function applyActivate(#[LiveArg] bool $activate, #[LiveArg] string $reason): void
    {
        if (!$this->security->isGranted('permission', [
            'permission' => 'role.edit',
            'role' => $this->role->code,
        ])) {
            throw new AccessDeniedException();
        }

        $user = $this->getSecurityUser();

        try {
            $this->manager->activate($this->role, $user->email, $reason, $activate);
            $this->flash('success', $activate ? 'Role activated.' : 'Role deactivated.');
        } catch (ActivateException|RoleNotFoundException|NotBlankException $e) {
            $this->flash('info', $e->getMessage() ?: 'No changes were made');
        }
    }
}