<?php
declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Form\Role;

use Atlas\SecurityManagerBundle\Contract\FormInterface;
use Atlas\SecurityManagerBundle\Contract\FormRepositoryInterface;
use Atlas\SecurityManagerBundle\Dto\Role\RoleDto;
use Atlas\SecurityManagerBundle\Entity\Role\Permission;
use Doctrine\ORM\EntityRepository;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\HiddenType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;

final class RoleForm extends AbstractType
{
    public function __construct(private readonly FormRepositoryInterface $forms)
    {

    }
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $includeReason = (bool) ($options['include_reason'] ?? false);
        $isEdit = (bool) ($options['is_edit'] ?? false);

        if ($isEdit) {
            $builder->add('code', HiddenType::class);
            $builder->add('id', HiddenType::class);
        }
        else {
            $builder->add('code', TextType::class, [
                'label' => 'Code',
                'required' => true,
                'empty_data' => '',
                'attr' => ['maxlength' => 25],
            ]);
        }

        $builder
            ->add('name', TextType::class, [
                'label' => 'Name',
                'required' => true,
                'empty_data' => '',
                'attr' => ['maxlength' => 50],
            ])
            ->add('permissions', EntityType::class, [
                'class' => Permission::class,
                'label' => 'Permissions',
                'required' => false,
                'multiple' => true,
                'expanded' => true,
                'choice_label' => static fn(Permission $p) => sprintf('%s [%s]', $p->name, $p->code),
                'choice_attr' => static fn (Permission $p) => [
                    'data-description' => $p->description,
                    'data-location-restricted' => $p->location_restricted ? '1' : '0',
                ],
                'query_builder' => static fn (EntityRepository $er) => $er->createQueryBuilder('p')
                    ->orderBy('p.code', 'ASC'),
            ]);
        $forms = $this->forms->findAllActive();
        if($forms !== []) {
            $builder->
            add('forms', ChoiceType::class, [
                'label' => 'Forms',
                'required' => false,
                'multiple' => true,
                'expanded' => true,

                // list of FormInterface objects
                'choices' => $forms,

                // stable identity for submitted values
                'choice_value' => static function (?FormInterface $f): ?string {
                    if($f === null) return null;
                    else return (string) $f->id;
                },

                'choice_label' => static function (FormInterface $f): string {
                    return sprintf('%s (%s)', $f->name, $f->code);
                },

                'choice_attr' => static function (FormInterface $f): array {
                    // keep it safe: these are methods on the interface you control
                    return [
                        'data-code' => $f->code,
                    ];
                },
            ]);
        }

        if ($includeReason) {
            $builder->add('reason', TextareaType::class, [
                'label' => 'Reason',
                'required' => true,
                'empty_data' => '',
                'attr' => ['rows' => 3, 'maxlength' => 255],
            ]);
        }
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => RoleDto::class,
            'include_reason' => false,
            'is_edit' => false,
        ]);
    }
}