<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Form\User;

use Atlas\SecurityManagerBundle\Contract\LocationInterface;
use Atlas\SecurityManagerBundle\Contract\LocationRepositoryInterface;
use Atlas\SecurityManagerBundle\Dto\User\UserRoleDto;
use Atlas\SecurityManagerBundle\Entity\Role\Role;
use Atlas\SecurityManagerBundle\Repository\Role\RoleRepository;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\Exception\AccessException;
use Symfony\Component\OptionsResolver\OptionsResolver;

final  class UserRoleForm extends AbstractType
{
    /**
     * @param RoleRepository $role_repository
     * @param LocationRepositoryInterface $locations
     */
    public function __construct(
        private readonly RoleRepository $role_repository,
        private readonly LocationRepositoryInterface $locations,
    )
    {}


    /**
     * @param FormBuilderInterface $builder
     * @param array $options
     * @return void
     */
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add('role', EntityType::class, [
                'class' => Role::class,
                'choice_label' => static function (Role $r): string {
                    return trim(sprintf('%s (%s)', $r->name, $r->code));
                },
                'query_builder' => fn() => $this->role_repository->createQueryBuilder('r')->orderBy('r.name', 'ASC'),
                'placeholder' => 'Select role',
                'required' => true,
            ]);

        $locations = $this->locations->findAll();

        $builder->add('location', ChoiceType::class, [
            'required' => false,
            'placeholder' => 'All locations', // null means global
            'choices' => $locations,

            // Symfony may call with null during normalisation
            'choice_value' => static function (?LocationInterface $l): ?string {
                if($l === null) return null;
                else return (string) $l->id;
            },

            'choice_label' => static fn (LocationInterface $l): string
            => trim(sprintf('%s (%s)', $l->name, $l->lid)),

            // optional but handy if you want it in DOM
            'choice_attr' => static fn (LocationInterface $l): array => [
                'data-lid' => $l->lid,
            ]
        ])->add('reason', TextareaType::class, [
                'empty_data' => '',
                'trim' => true,
                'attr' => [
                    'rows' => 1,
                ],
            ]
        );
    }


    /**
     * @param OptionsResolver $resolver
     * @return void
     * @throws AccessException
     */
    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => UserRoleDto::class,
        ]);
    }
}