<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Listener;

use Atlas\SecurityManagerBundle\Entity\Audit\Login;
use Atlas\SecurityManagerBundle\Entity\Audit\Password;
use Atlas\SecurityManagerBundle\Exception\User\InternalUserException;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\SecurityManagerBundle\Repository\Audit\LoginRepository;
use Atlas\SecurityManagerBundle\Repository\User\UserRepository;
use Atlas\SecurityManagerBundle\Security\User as SecurityUser;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\EventDispatcher\Attribute\AsEventListener;
use Symfony\Component\HttpFoundation\Exception\SessionNotFoundException;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\Security\Http\Event\InteractiveLoginEvent;
use Symfony\Component\Security\Http\Event\LoginFailureEvent;
use Symfony\Component\Security\Http\SecurityEvents;
use function count;

final readonly class MonitorListener
{
    public function __construct(
        private EntityManagerInterface $entity_manager,
        private LoginRepository $logins,
        private UserRepository $users,
        private UserPasswordHasherInterface $passwords,
    ) {}

    /**
     * @throws InternalUserException
     * @throws NotBlankException
     * @throws SessionNotFoundException
     */
    #[AsEventListener(event: SecurityEvents::INTERACTIVE_LOGIN, dispatcher: 'security.event_dispatcher.main')]
    public function onSecurityInteractiveLogin(InteractiveLoginEvent $event): void
    {
        $securityUser = $event->getAuthenticationToken()->getUser();

        if (!$securityUser instanceof SecurityUser) {
            return;
        }

        $user = $this->users->findOneByUsernameOrEmail($securityUser->username);
        if (!$user) {
            return; // nothing to do
        }

        // Non-blocking UI warning if duplicates exist
        $all = $this->users->findAllByUsernameOrEmail($securityUser->username);
        if (count($all) > 1) {
            $event->getRequest()->getSession()->getFlashBag()->add(
                'warning',
                'We found more than one account matching your username/email. You were signed in to your primary account; contact support if this seems wrong.'
            );
        }

        $log = new Login();
        $log->logLoginSuccess($user);
        $this->entity_manager->persist($log);
        $this->entity_manager->flush();

        if ($securityUser->internal) {
            if (
                $user->firstname !== $securityUser->firstname ||
                $user->lastname  !== $securityUser->lastname  ||
                $user->email     !== $securityUser->email
            ) {
                $user->ldapSyncUser(
                    firstname: $securityUser->firstname,
                    lastname: $securityUser->lastname,
                    email: $securityUser->email
                );
            }

            $this->entity_manager->persist($user);
            $this->entity_manager->flush();

            // Guard: request param may be null
            $passwordToCheck = (string) $event->getRequest()->request->get('_password', '');
            if ($passwordToCheck !== '' && !$this->passwords->isPasswordValid($securityUser, $passwordToCheck)) {
                $password = new Password(
                    $user,
                    $this->passwords->hashPassword($securityUser, $passwordToCheck),
                    'LDAP'
                );
                $this->entity_manager->persist($password);
                $this->entity_manager->flush();
            }
        }
    }

    #[AsEventListener(event: LoginFailureEvent::class, dispatcher: 'security.event_dispatcher.main')]
    public function onSecurityLoginFailure(LoginFailureEvent $event): void
    {
        $securityUser = $event->getPassport()?->getUser();
        $message      = $event->getException()->getMessage();

        if (!$securityUser instanceof SecurityUser) {
            return;
        }

        // Warn the person on the login page if duplicates exist
        $dupes = $this->users->findAllByUsernameOrEmail($securityUser->username);
        if (count($dupes) > 1) {
            $event->getRequest()->getSession()->getFlashBag()->add(
                'warning',
                'There are multiple accounts with that username/email. If you cannot sign in, please contact support to merge accounts.'
            );
        }

        $user = $this->users->findOneByUsernameOrEmail($securityUser->username);
        if (!$user) {
            return; // unknown user: nothing to log against
        }

        $log = $this->logins->findLastLogin($user->id);

        if (!$log) {
            $count = 0;
        } else {
            $count = $log->failed_count + 1;

            if ($count === 10) {
                $user->lock('system', 'User reached 10 failed logins', true);
                $this->entity_manager->persist($user);
                $this->entity_manager->flush();
            }
        }

        $log = new Login();
        $log->logLoginFail($user, count: $count, reason: $message);

        $this->entity_manager->persist($log);
        $this->entity_manager->flush();
    }
}
