<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Repository\Role;

use Atlas\SecurityManagerBundle\Entity\Role\Permission;
use Atlas\SecurityManagerBundle\Entity\Role\RolePermission;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;
use Psr\Cache\InvalidArgumentException;
use Symfony\Contracts\Cache\CacheInterface;
use Symfony\Contracts\Cache\ItemInterface;

/**
 * @extends ServiceEntityRepository<RolePermission>
 */
class RolePermissionRepository extends ServiceEntityRepository
{
    private const string CACHE_KEY = 'shared_role_permissions_map';
    private const int TTL = 86400;

    public function __construct(
        ManagerRegistry $registry,
        private readonly CacheInterface $cache)
    {
        parent::__construct($registry, RolePermission::class);
    }

    /**
     * @param bool $forceRebuild
     * @return array
     * @throws InvalidArgumentException
     */
    public function getPermissionMap(bool $forceRebuild = false) : array
    {

        if($forceRebuild) $this->clearPermissionMapCache();

        return $this->cache->get(self::CACHE_KEY, function(ItemInterface $item): array {

            $item->expiresAfter(self::TTL);

            $permissions = $this->createQueryBuilder('rp')
                ->select('r.code AS role_code', 'p.code AS permission_code')
                ->join('rp.role', 'r')
                ->join('rp.permission', 'p')
                ->where('r.active = true')
                ->andWhere('p.active = true')
                ->orderBy('r.code', 'ASC')
                ->addOrderBy('p.code', 'ASC')
                ->getQuery()
                ->getScalarResult();

            $map = array();

            foreach ($permissions as $row) {
                $map[$row['role_code']][] = $row['permission_code'];
            }

            return $map;
        });
    }

    /**
     * @param int $roleId
     * @return array
     */
    public function findPermissionsForRoleId(int $roleId): array
    {
        return $this->getEntityManager()
            ->createQueryBuilder()
            ->select('p')
            ->from(Permission::class, 'p')
            ->innerJoin(
                RolePermission::class,
                'rp',
                'WITH',
                'rp.permission = p'
            )
            ->where('IDENTITY(rp.role) = :rid')
            ->setParameter('rid', $roleId)
            ->orderBy('p.name', 'ASC')
            ->addOrderBy('p.code', 'ASC')
            ->getQuery()
            ->getResult();
    }

    /**
     * Manually clear the permissions map cache.
     *
     * @throws InvalidArgumentException
     */
    public function clearPermissionMapCache(): void
    {
        $this->cache->delete(self::CACHE_KEY);
    }
}
