<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Security\Authenticator;

use Atlas\SecurityManagerBundle\Security\User as SecurityUser;
use Psr\Log\LoggerInterface;
use Stringable;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\Ldap\Exception\ConnectionException;
use Symfony\Component\Ldap\Exception\LdapException;
use Symfony\Component\Ldap\LdapInterface;
use Symfony\Component\Ldap\Security\LdapBadge;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Exception\BadCredentialsException;
use Symfony\Component\Security\Core\User\PasswordUpgraderInterface;
use Symfony\Component\Security\Core\User\UserProviderInterface;
use Symfony\Component\Security\Http\Authenticator\AbstractLoginFormAuthenticator;
use Symfony\Component\Security\Core\Exception\CustomUserMessageAuthenticationException;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\CsrfTokenBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\PasswordUpgradeBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\RememberMeBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\UserBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Credentials\PasswordCredentials;
use Symfony\Component\Security\Http\Authenticator\Passport\Passport;
use Symfony\Component\Security\Http\Util\TargetPathTrait;
use Symfony\Component\Security\Http\SecurityRequestAttributes;

final class AtlasAdaptor extends AbstractLoginFormAuthenticator
{
    use TargetPathTrait;

    public const string LOGIN_ROUTE = 'shared_login';

    /** @var class-string<LdapInterface> */
    private readonly string $ldap_class;

    //automatically define class private variables from constructor
    public function __construct(
        private readonly UrlGeneratorInterface $url_generator,
        private readonly UserProviderInterface $user_provider,
        private readonly LdapInterface $ldap,
        private readonly LoggerInterface $logger)
    {
        $this->ldap_class = get_class($ldap);
    }

    public function authenticate(Request $request): Passport
    {
        $jsCapable = (string) $request->request->get('js_capable', '0');
        if ($jsCapable !== '1') {
            $this->logger?->info('Blocked login: missing js_capable', [
                'ip' => $request->getClientIp(),
                'ua' => $request->headers->get('User-Agent'),
            ]);

            throw new CustomUserMessageAuthenticationException(
                'Your browser is not supported. Please enable JavaScript or contact your support team.'
            );
        }

        $credentials = $this->getCredentials($request);

        $user = $this->user_provider->loadUserByIdentifier($credentials['username']);

        $userBadge = new UserBadge($credentials['username'], fn($id) => $user);

        $csrfBadge = new CsrfTokenBadge('authenticate', $credentials['csrf_token']);

        $passport = new Passport(
            $userBadge,
            new PasswordCredentials($credentials['password']),
            [ $csrfBadge, new RememberMeBadge() ]
        );



        if ($this->user_provider instanceof PasswordUpgraderInterface) {
            $passport->addBadge(new PasswordUpgradeBadge($credentials['password'], $this->user_provider));
        }

        $user = $userBadge->getUser();

        if($user instanceof SecurityUser) {
            try {
                if (!empty($user->dn)) {
                    $passport->addBadge(
                        new LdapBadge(
                            $this->ldap_class,
                            $user->dn
                        )
                    );
                }
            } catch (ConnectionException|LdapException) {
                //only fall back if a connection exception can then use db password
            }
        }

        return $passport;
    }

    public function onAuthenticationSuccess(Request $request, TokenInterface $token, string $firewallName): ?Response
    {
        if ($targetPath = $this->getTargetPath($request->getSession(), $firewallName)) {
            return new RedirectResponse($targetPath);
        }

        return new RedirectResponse($this->url_generator->generate('app_dashboard'));
    }

    protected function getLoginUrl(Request $request): string
    {
        return $this->url_generator->generate(self::LOGIN_ROUTE);
    }

    private function getCredentials(Request $request): array
    {
        $credentials = array();

        $credentials['username'] = $request->request->get('_username');
        $credentials['username'] = mb_trim($credentials['username'], encoding: 'UTF-8');

        if ('' === $credentials['username']) {
            throw new BadCredentialsException('The username or email must be a non-empty string');
        }

        $request->getSession()->set(SecurityRequestAttributes::LAST_USERNAME, $credentials['username']);

        $credentials['password'] = $request->request->get('_password');

        if (!is_string($credentials['password']) && !$credentials['password'] instanceof Stringable) {
            throw new BadRequestHttpException(
                sprintf(
                    'The password must be a string, "%s" given.',
                    gettype($credentials['password'])
                )
            );
        }

        if ('' === (string) $credentials['password']) {
            throw new BadCredentialsException('The password must be a non-empty string.');
        }

        $credentials['csrf_token'] = $request->request->get('_csrf_token');

        if (!is_string($credentials['csrf_token'] ?? '') && !$credentials['csrf_token'] instanceof Stringable) {
            throw new BadRequestHttpException(
                sprintf(
                    'The CSRF Token must be a string, "%s" given.',
                    gettype($credentials['csrf_token'])
                )
            );
        }

        return $credentials;
    }
}
