<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Service\Role;

use Atlas\SecurityManagerBundle\Dto\Role\RoleDto;
use Atlas\SecurityManagerBundle\Entity\Role\Role;
use Atlas\SecurityManagerBundle\Entity\Role\RoleForm;
use Atlas\SecurityManagerBundle\Entity\Role\RolePermission;
use Atlas\SecurityManagerBundle\Exception\NoUpdateRequiredException;
use Atlas\SecurityManagerBundle\Exception\Role\RoleException;
use Atlas\SecurityManagerBundle\Exception\Role\RoleNotFoundException;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\SecurityManagerBundle\Repository\Role\RoleFormRepository;
use Atlas\SecurityManagerBundle\Repository\Role\RolePermissionRepository;
use Atlas\SecurityManagerBundle\Repository\Role\RoleRepository;
use Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use Doctrine\ORM\Cache\LockException;
use Doctrine\ORM\EntityManagerInterface;
use Modules\Paginator\Service\Paginator;
use Psr\Cache\InvalidArgumentException;
use Throwable;

final readonly class RoleManager
{
    /**
     * @param EntityManagerInterface $entity_manager
     * @param RolePermissionRepository $permissions
     * @param RoleFormRepository $forms
     * @param RoleRepository $roles
     * @param Paginator $paginator
     */
    public function __construct(
        private EntityManagerInterface $entity_manager,
        private RolePermissionRepository $permissions,
        private RoleFormRepository $forms,
        private RoleRepository $roles,
        private Paginator $paginator
    )
    {
    }

    /**
     * @param array $options expected keys: c_f, c_o, c_d, c_s, p
     * @param int $perPage
     * @return array
     */
    public function getRolesPage(array $options, int $perPage = 50): array
    {
        $query = $this->roles->findAllQueryBuilder(
            filter: $options['c_f'] ?? null,
            sort: $options['c_o'] ?? null,
            direction: $options['c_d'] ?? null,
            search: $options['c_s'] ?? null
        );

        $page = (int)($options['p'] ?? 1);

        if ($page < 1) {
            $page = 1;
        }

        $perPage = max(1, $perPage);

        return $this->paginator->paginate($query, $page, $perPage);
    }

    /**
     * @param RoleDto $dto
     * @param string $actionBy
     * @return Role
     */
    public function createRole(RoleDto $dto, string $actionBy): Role
    {
        $role = new Role($dto->code, $dto->name, $actionBy);

        try {
            $this->entity_manager->persist($role);
            $this->entity_manager->flush();

            $reason = $dto->reason !== '' ? $dto->reason : 'Initial creation of role';

            $permChanged = $this->syncRolePermissions($role, $dto->permissions ?? [], $actionBy, $reason);
            $formChanged = $this->syncRoleForms($role, $dto->forms ?? [], $actionBy, $reason);

            $this->entity_manager->flush();

            if ($permChanged) {
                $this->permissions->clearPermissionMapCache();
            }
            if ($formChanged) {
                $this->forms->clearPermissionMapCache();
            }

        } catch (UniqueConstraintViolationException $e) {
            throw new RoleException('Failed to create role due to unique constraint violation', 0, $e);
        } catch (Throwable $e) {
            throw new RoleException('Failed to create role: ' . $e->getMessage(), 0, $e);
        }

        return $role;
    }

    /**
     * @param int|Role $role
     * @param string $actionBy
     * @param string $reason
     * @param bool $activate
     * @return Role
     * @throws RoleNotFoundException
     * @throws LockException
     * @throws NotBlankException
     * @throws InvalidArgumentException
     */
    public function activate(
        int|Role $role,
        string $actionBy,
        string $reason,
        bool $activate = true
    ): Role
    {

        if (is_int($role)) {
            $role = $this->roles->findOrThrow($role);
        }

        // Keep original exception behaviour for the main entity
        $role->activate($actionBy, $reason, $activate);

        $this->permissions->clearPermissionMapCache();
        $this->forms->clearPermissionMapCache();

        // Unlock only affects this node, no recursion
        $this->entity_manager->flush();
        return $role;
    }


    /**
     * @param int|Role $role
     * @return RoleDto
     * @throws RoleNotFoundException
     */
    public
    function getRoleDto(
        int|Role $role
    ): RoleDto
    {
        if (is_int($role)) {
            $role = $this->roles->findOrThrow($role);
        }

        $dto = new RoleDto();
        $dto->id = $role->id;
        $dto->name = $role->name;
        $dto->code = $role->code;

        $dto->permissions = $this->permissions->findPermissionsForRoleId($role->id);
        $dto->forms = $this->forms->findFormsForRoleId($role->id);

        return $dto;
    }

    /**
     * @param int|Role $role
     * @param RoleDto $dto
     * @param string $actionBy
     * @return Role
     * @throws InvalidArgumentException
     */
    public function editRole(int|Role $role, RoleDto $dto, string $actionBy): Role
    {
        if (is_int($role)) {
            $role = $this->roles->findOrThrow($role);
        }

        // If include_reason=false you must not require reason.
        // But your Role::update() requires it, so for edit screens this must be present.
        $reason = $dto->reason;

        $changed = false;

        try {
            $role->update($actionBy, $dto->reason, name: $dto->name);
            $changed = true;
        } catch (NoUpdateRequiredException) {
            // ignore — permissions/forms may still change
        }

        $permChanged = $this->syncRolePermissions($role, $dto->permissions ?? [], $actionBy, $reason);
        $formChanged = $this->syncRoleForms($role, $dto->forms ?? [], $actionBy, $reason);

        $changed = $changed || $permChanged || $formChanged;

        if (!$changed) {
            throw NoUpdateRequiredException::forId('Role', $role->id);
        }

        $this->entity_manager->flush();

        if ($permChanged) {
            $this->permissions->clearPermissionMapCache();
        }
        if ($formChanged) {
            $this->forms->clearPermissionMapCache();
        }

        return $role;
    }

    /**
     * @param Role $role
     * @param array $newPermissions Permission[] from DTO
     * @param string $actionBy
     * @param string $reason
     * @return bool changed?
     */
    private function syncRolePermissions(Role $role, array $newPermissions, string $actionBy, string $reason): bool
    {
        /** @var RolePermission[] $existing */
        $existing = $this->permissions->findBy(['role' => $role]);

        $existingByPermissionId = [];
        foreach ($existing as $rp) {
            $existingByPermissionId[(int) $rp->permission->id] = $rp;
        }

        $newByPermissionId = [];
        foreach ($newPermissions as $p) {
            $newByPermissionId[(int) $p->id] = $p;
        }

        $changed = false;

        // removals (audit needs preDelete)
        foreach ($existingByPermissionId as $pid => $rp) {
            if (!isset($newByPermissionId[$pid])) {
                $rp->preDelete($actionBy, $reason);
                $this->entity_manager->remove($rp);
                $changed = true;
            }
        }

        // additions (constructor requires role/permission/action/reason)
        foreach ($newByPermissionId as $pid => $p) {
            if (!isset($existingByPermissionId[$pid])) {
                $this->entity_manager->persist(
                    new RolePermission($role, $p, $actionBy, $reason)
                );
                $changed = true;
            }
        }

        return $changed;
    }

    /**
     * @param Role $role
     * @param array $newForms FormMeta[] from DTO
     * @param string $actionBy
     * @param string $reason
     * @return bool changed?
     */
    private function syncRoleForms(Role $role, array $newForms, string $actionBy, string $reason): bool
    {
        /** @var RoleForm[] $existing */
        $existing = $this->forms->findBy(['role' => $role]);

        $existingByFormId = [];
        foreach ($existing as $rf) {
            $existingByFormId[(int) $rf->form->id] = $rf;
        }

        $newByFormId = [];
        foreach ($newForms as $f) {
            $newByFormId[(int) $f->id] = $f;
        }

        $changed = false;

        foreach ($existingByFormId as $fid => $rf) {
            if (!isset($newByFormId[$fid])) {
                $rf->preDelete($actionBy, $reason);
                $this->entity_manager->remove($rf);
                $changed = true;
            }
        }

        foreach ($newByFormId as $fid => $f) {
            if (!isset($existingByFormId[$fid])) {
                $this->entity_manager->persist(
                    new RoleForm($role, $f, $actionBy, $reason)
                );
                $changed = true;
            }
        }

        return $changed;
    }
}
