<?php
declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Service\Security;

use Atlas\SecurityManagerBundle\Exception\Security\TokenException;
use DomainException;
use Firebase\JWT\BeforeValidException;
use Firebase\JWT\ExpiredException;
use Firebase\JWT\JWT;
use Firebase\JWT\Key;
use Firebase\JWT\SignatureInvalidException;
use InvalidArgumentException;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use TypeError;
use UnexpectedValueException;
use function file_get_contents;

final readonly class JwtTokenChecker
{
    public function __construct(
        #[Autowire('%shared.base.specs_dir%/keys')] private string $public_keys_dir,
        #[Autowire('%shared.jwt.iss%')] private string $jwt_iss,
        #[Autowire('%shared.jwt.aud%')] private string $jwt_aud,
        private int $leeway_seconds = 60,
    ) {}

    /**
     * @param string $jwt
     * @return array<string,mixed>
     * @throws TokenException
     * @throws DomainException
     * @throws BeforeValidException
     * @throws ExpiredException
     * @throws SignatureInvalidException
     * @throws InvalidArgumentException
     * @throws TypeError
     * @throws UnexpectedValueException
     */
    public function verify(string $jwt): array
    {
        $kid = $this->extractKid($jwt);

        // Strict allowlist: letters, digits, dot, underscore, hyphen
        if (!preg_match('/^[A-Za-z0-9._-]{1,64}$/', $kid)) {
            throw new TokenException('Invalid key format');
        }

        $pub_path = sprintf('%s/%s.pem', $this->public_keys_dir, $kid);
        $public_pem = file_get_contents($pub_path);
        if ($public_pem === false) {
            throw new TokenException('Unknown public key');
        }

        JWT::$leeway = $this->leeway_seconds;

        $decoded = JWT::decode($jwt, new Key($public_pem, 'ES256'));

        /** @var array<string,mixed> $claims */
        $claims = (array) $decoded;

        if (($claims['iss'] ?? null) !== $this->jwt_iss) {
            throw new TokenException('Bad issuer');
        }
        if (($claims['aud'] ?? null) !== $this->jwt_aud) {
            throw new TokenException('Bad audience');
        }

        return $claims;
    }

    /**
     * @param string $jwt
     * @return string
     * @throws TokenException
     */
    private function extractKid(string $jwt): string
    {
        $parts = explode('.', $jwt, 3);
        if (count($parts) < 2) {
            throw new TokenException('Malformed JWT');
        }
        $hdr_json = base64_decode(strtr($parts[0], '-_', '+/'), true);
        if ($hdr_json === false) {
            throw new TokenException('Bad header');
        }
        $hdr = json_decode($hdr_json, true);
        if (!is_array($hdr)) {
            throw new TokenException('Bad header');
        }
        $kid = $hdr['kid'] ?? null;
        if (!is_string($kid) || $kid === '') {
            throw new TokenException('Missing KID');
        }
        return $kid;
    }
}
