<?php declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Service\Security;

use Atlas\SecurityManagerBundle\Entity\Audit\Password as PasswordAudit;
use Atlas\SecurityManagerBundle\Entity\User\User;
use Atlas\SecurityManagerBundle\Exception\Security\InvalidClaimException;
use Atlas\SecurityManagerBundle\Exception\Validation\ValidationException;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\SecurityManagerBundle\Contract\EmailNotifierInterface;
use Atlas\SecurityManagerBundle\Repository\Audit\PasswordRepository;
use Atlas\SecurityManagerBundle\Repository\User\UserRepository;
use Atlas\SecurityManagerBundle\Security\Provider\UserProvider;
use DateMalformedStringException;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Firebase\JWT\ExpiredException;
use RuntimeException;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\PasswordHasher\Hasher\PasswordHasherFactoryInterface;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\Security\Core\Exception\UserNotFoundException;
use Symfony\Component\Security\Core\User\PasswordAuthenticatedUserInterface;
use Throwable;

/**
 * Orchestrates password reset operations (token validation + password set).
 */
final readonly class PasswordManager
{
    public function __construct(
        #[Autowire('%shared.base.url%')] private string $base_url,
        private PasswordPolicy $password_policy,
        private UserRepository $user_repository,
        private PasswordRepository $password_repository,
        private UserPasswordHasherInterface $user_hasher,
        private PasswordHasherFactoryInterface $hasher_factory,
        private JwtTokenChecker $jwt_checker,
        private JwtTokenGenerator $jwt_generator,
        private EntityManagerInterface $entity_manager,
        private UserProvider $user_provider,
        private EmailNotifierInterface $mailer
    ) {}

    /**
     * Validate a password reset token without mutating any state.
     *
     * @param string $token
     * @return User|string[] User on success, or error messages on failure.
     * @throws DateMalformedStringException
     */
    public function checkPasswordToken(string $token): User|array
    {
        try {
            /** @var array<string,mixed> $claims */
            $claims = $this->jwt_checker->verify($token);
        } catch (ExpiredException) {
            return ['Password token expired'];
        } catch (Throwable) {
            return ['Invalid token'];
        }

        // Purpose must be "pwd_set"
        if (($claims['purpose'] ?? null) !== 'pwd_set') {
            return ['Invalid token'];
        }

        // Subject (username/email)
        $username = trim((string) ($claims['sub'] ?? ''));
        if ($username === '') {
            return ['Invalid token'];
        }

        // Domain user
        $user = $this->user_repository->findOneByUsernameOrEmail($username);
        if (!$user instanceof User) {
            return ['User does not exist'];
        }

        // Token must be newer than the latest password change
        $iat = isset($claims['iat']) ? (int) $claims['iat'] : 0;
        if ($iat > 0) {
            $tokenIssuedAt = new DateTimeImmutable('@' . $iat);
            $latest = $this->password_repository->getLatestPassword($user->id);
            if ($latest !== null && $latest->changed > $tokenIssuedAt) {
                return ['Password already changed'];
            }
        }

        return $user;
    }

    /**
     * Validate the new password (policy + reuse) and persist it.
     * Assumes the token has already been validated via checkPasswordToken().
     *
     * @param User $user
     * @param string $newPassword
     * @return array<int,string> Empty on success; otherwise user-facing errors.
     * @throws NotBlankException
     * @throws RuntimeException
     * @throws UserNotFoundException
     * @throws DateMalformedStringException
     */
    public function resetPassword(User $user, string $newPassword): array
    {
        if ($user->is_locked) {
            return ['Your account is locked, please contact your study team for further support.'];
        }

        $policyErrors = $this->password_policy->validate($newPassword);
        if ($policyErrors !== []) {
            return $policyErrors;
        }

        // Hashing context via security user
        $securityUser = $this->user_provider->loadUserByIdentifier($user->username);

        // Reuse check vs. recent N hashes
        $hasher = $this->hasher_factory->getPasswordHasher($securityUser);
        if (array_any(
            $this->password_repository->getRecentPasswordHashes($user->id),
            static fn(string $oldHash): bool => $hasher->verify($oldHash, $newPassword)
        )) {
            return ['You have used this password recently. Choose a new password you have not used in your last 6 changes.'];
        }

        if (! $securityUser instanceof PasswordAuthenticatedUserInterface) {
            throw new RuntimeException('Loaded user must implement PasswordAuthenticatedUserInterface.');
        }

        // Hash + audit trail
        $newHash = $this->user_hasher->hashPassword($securityUser, $newPassword);
        $audit = new PasswordAudit($user, $newHash, $user->email);

        $this->entity_manager->persist($audit);
        $this->entity_manager->flush();

        // Best-effort validation toggle
        if (!$user->is_validated) {
            try {
                $user->validate();
                $this->entity_manager->flush();
            } catch (ValidationException) {
                // ignore
            }
        }

        return [];
    }

    /**
     * @param string $identifier
     * @return bool
     * @throws NotBlankException
     * @throws InvalidClaimException
     */
    public function request(string $identifier): bool
    {
        $identifier = mb_trim($identifier, encoding: 'UTF-8');

        if($identifier === '') {
            throw new NotBlankException('Username or email cannot be blank');
        }

        $user = $this->user_repository->findOneByUsernameOrEmail($identifier);

        if (! $user) {
            return true; //do not leak if user doesn't exist
        }

        // Generate single-purpose JWT (your generator already does this)
        $token = $this->jwt_generator->issuePasswordReset($user->username);

        //if we get here we can issue the password token
        $data = [
            'user_email' => $user->email,
            'username' => $user->username,
            'firstname' => $user->firstname,
            'lastname' => $user->lastname,
            'link' => sprintf('%s/password/%s', $this->base_url,  $token)
        ];

        $this->mailer->send('FORGOT', $user->email, $data);

        return true;
    }
}
