<?php
declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Twig\Components\Role;

use Atlas\SecurityManagerBundle\Dto\Role\RoleDto;
use Atlas\SecurityManagerBundle\Exception\NoUpdateRequiredException;
use Atlas\SecurityManagerBundle\Exception\Role\RoleException;
use Atlas\SecurityManagerBundle\Exception\Role\RoleNotFoundException;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\SecurityManagerBundle\Form\Role\RoleForm as Form;
use Atlas\SecurityManagerBundle\Service\Role\RoleManager;
use Atlas\SecurityManagerBundle\Twig\Components\Concern\FlashHelperTrait;
use Atlas\SecurityManagerBundle\Twig\Components\Concern\SecurityUserHelperTrait;
use InvalidArgumentException;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\Form\Exception\LogicException;
use Symfony\Component\Form\Exception\RuntimeException;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Exception\UnprocessableEntityHttpException;
use Symfony\Component\OptionsResolver\Exception\InvalidOptionsException;
use Symfony\Component\Routing\Exception\InvalidParameterException;
use Symfony\Component\Routing\Exception\MissingMandatoryParametersException;
use Symfony\Component\Routing\Exception\RouteNotFoundException;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Security\Core\Exception\AccessDeniedException;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveAction;
use Symfony\UX\LiveComponent\Attribute\LiveProp;
use Symfony\UX\LiveComponent\ComponentToolsTrait;
use Symfony\UX\LiveComponent\ComponentWithFormTrait;
use Symfony\UX\LiveComponent\DefaultActionTrait;

#[AsLiveComponent(
    name: 'role_form',
    template: '@SecurityManager/components/role/role_form.html.twig'
)]
final class RoleFormComponent
{
    use DefaultActionTrait;
    use ComponentToolsTrait;
    use ComponentWithFormTrait;
    use FlashHelperTrait;
    use SecurityUserHelperTrait;

    public function __construct(
        private readonly FormFactoryInterface $forms,
        private readonly RoleManager $manager,
        private readonly UrlGeneratorInterface $urls,
        private readonly Security $security,
        private readonly RequestStack $requests,
    ) {}

    #[LiveProp(writable: true)]
    public bool $include_reason = false;

    #[LiveProp]
    public ?int $role_id = null;

    public bool $is_edit { get => $this->role_id !== null; }

    /**
     * @throws InvalidOptionsException
     * @throws RoleNotFoundException
     */
    protected function instantiateForm(): FormInterface
    {
        if (! $this->is_edit) {
            $dto = new RoleDto();
        } else {
            $dto = $this->manager->getRoleDto($this->role_id);
        }

        $dto->include_reason = $this->include_reason;

        return $this->forms->createNamed(
            'form',
            Form::class,
            $dto,
            [
                'include_reason' => $this->include_reason,
                'csrf_protection' => true,
                'is_edit' => $this->is_edit,
            ]
        );
    }

    protected function shouldSubmitFormOnRender(): bool
    {
        return false;
    }

    /**
     * @throws AccessDeniedException
     * @throws InvalidArgumentException
     * @throws LogicException
     * @throws RuntimeException
     * @throws UnprocessableEntityHttpException
     * @throws InvalidParameterException
     * @throws MissingMandatoryParametersException
     * @throws RouteNotFoundException
     */
    #[LiveAction]
    public function save(): ?RedirectResponse
    {
        $this->submitForm();

        if (! $this->form->isValid()) {
            return null;
        }

        /** @var RoleDto $dto */
        $dto = $this->form->getData();

        $user = $this->getSecurityUser();

        $role = null;

        if ($this->is_edit) {
            try {
                $role = $this->manager->editRole($this->role_id, $dto, $user->email);
                $this->persistFlash('success', 'Role edited successfully');
            } catch (RoleNotFoundException|NoUpdateRequiredException|NotBlankException $e) {
                $this->flash('warning', $e->getMessage() ?: 'Could not edit role');
            }
        } else {
            try {
                $role = $this->manager->createRole($dto, $user->email);
                $this->persistFlash('success', 'Role added successfully');
            } catch (RoleException $e) {
                $this->flash('warning', $e->getMessage() ?: 'Could not add role');
            }
        }

        if ($role !== null) {
            return new RedirectResponse(
                $this->urls->generate('shared_role_view', ['id' => $role->id])
            );
        }

        return null;
    }
}