<?php declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Twig\Components\Security;

use Atlas\SecurityManagerBundle\Entity\User\User;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\SecurityManagerBundle\Service\Security\PasswordManager;
use Atlas\SecurityManagerBundle\Service\Security\PasswordPolicy;
use Atlas\SecurityManagerBundle\Twig\Components\Concern\FlashHelperTrait;
use DateMalformedStringException;
use RuntimeException;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Security\Core\Exception\UserNotFoundException;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveAction;
use Symfony\UX\LiveComponent\Attribute\LiveProp;
use Symfony\UX\LiveComponent\ComponentToolsTrait;
use Symfony\UX\LiveComponent\ComponentWithFormTrait;
use Symfony\UX\LiveComponent\DefaultActionTrait;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\Form\FormFactoryInterface;
use Atlas\SecurityManagerBundle\Form\Security\PasswordResetForm as Form;
use Symfony\Component\HttpFoundation\RequestStack;

#[AsLiveComponent(
    name: 'password_reset_form',
    template: '@SecurityManager/components/security/password_reset_form.html.twig')
]
final class PasswordResetFormComponent
{
    use DefaultActionTrait;
    use ComponentWithFormTrait;
    use ComponentToolsTrait;
    use FlashHelperTrait;

    public function __construct(
        private readonly FormFactoryInterface $form_factory,
        private readonly PasswordManager $password_manager,
        private readonly PasswordPolicy $password_policy,
        private readonly RequestStack $requests,
        private readonly UrlGeneratorInterface $urls
    ) {}

    #[LiveProp]
    public ?User $user = null;


    #[LiveProp(writable: true)]
    public string $password = '';

    #[LiveProp(writable: true)]
    public string $confirm_password = '';

    /** Display-only state */
    #[LiveProp]
    public array $policy_errors = [];

    #[LiveProp]
    public array $flow_errors = [];

    #[LiveProp]
    public bool $success = false;

    protected function instantiateForm(): FormInterface
    {
        return $this->form_factory->createNamed(
            'form',
            Form::class,
            [
                'password' => $this->password,
                'confirm_password' => $this->confirm_password,
            ],
            [
                'csrf_protection' => true,
            ]
        );
    }

    /**
     * @return bool
     */
    protected function shouldSubmitFormOnRender(): bool
    {
        return false;
    }

    #[LiveAction]
    public function validatePolicy(): void
    {
        // Called on input change (optional—hook from template if desired)
        $errors = $this->password_policy->validate($this->password);
        if ($this->password !== '' && $this->confirm_password !== '' && $this->password !== $this->confirm_password) {
            $errors[] = 'Passwords don’t match. Make sure both fields are the same.';
        }
        $this->policy_errors = $errors;
    }

    /**
     * @throws \InvalidArgumentException
     */
    #[LiveAction]
    public function save(): ?RedirectResponse
    {
        $this->submitForm();

        $errors = $this->password_policy->validate($this->password);

        if ($this->password === '' || $this->password !== $this->confirm_password) {
            $errors[] = 'Passwords don’t match. Make sure both fields are the same.';
        }
        $this->policy_errors = $errors;

        if ($errors !== []) {
            $this->success = false;
            return null;
        }

        $flow = [];

        try {
            $flow = $this->password_manager->resetPassword($this->user, $this->password);
        } catch (NotBlankException|UserNotFoundException $e) {
            $this->flash('warning', $e->getMessage());
        } catch (RuntimeException|DateMalformedStringException $e) {
            $this->flash('warning', 'Error while resetting password');
        }

        $this->flow_errors = $flow;

        $this->success = ($flow === []);

        if ($this->success) {
            $this->persistFlash('success', 'Your password has been changed. You can now sign in.');

            return new RedirectResponse(
                $this->urls->generate('shared_login')
            );

        }

        return null;
    }
}
