<?php

namespace Atlas\SecurityManagerBundle\Twig\Components\User;

use Atlas\SecurityManagerBundle\Dto\User\UserDto;
use Atlas\SecurityManagerBundle\Exception\NoUpdateRequiredException;
use Atlas\SecurityManagerBundle\Exception\User\InternalUserException;
use Atlas\SecurityManagerBundle\Exception\User\UserNotFoundException;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\SecurityManagerBundle\Form\User\ExternalUserForm as Form;
use Atlas\SecurityManagerBundle\Service\User\UserManager;
use Atlas\SecurityManagerBundle\Twig\Components\Concern\FlashHelperTrait;
use Atlas\SecurityManagerBundle\Twig\Components\Concern\SecurityUserHelperTrait;
use InvalidArgumentException;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\Form\Exception\LogicException;
use Symfony\Component\Form\Exception\RuntimeException;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Exception\UnprocessableEntityHttpException;
use Symfony\Component\OptionsResolver\Exception\InvalidOptionsException;
use Symfony\Component\Routing\Exception\InvalidParameterException;
use Symfony\Component\Routing\Exception\MissingMandatoryParametersException;
use Symfony\Component\Routing\Exception\RouteNotFoundException;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Security\Core\Exception\AccessDeniedException;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveAction;
use Symfony\UX\LiveComponent\Attribute\LiveProp;
use Symfony\UX\LiveComponent\ComponentToolsTrait;
use Symfony\UX\LiveComponent\ComponentWithFormTrait;
use Symfony\UX\LiveComponent\DefaultActionTrait;

#[AsLiveComponent(
    name: 'user_external_form',
    template: '@SecurityManager/components/user/external_user_form.html.twig'
)]
final class ExternalUserFormComponent
{
    use DefaultActionTrait;
    use ComponentWithFormTrait;
    use ComponentToolsTrait;
    use FlashHelperTrait;
    use SecurityUserHelperTrait;

    /**
     * @param FormFactoryInterface $forms
     * @param UserManager $manager
     * @param UrlGeneratorInterface $urls
     * @param RequestStack $requests
     * @param Security $security
     */
    public function __construct(
        private readonly FormFactoryInterface $forms,
        private readonly UserManager $manager,
        private readonly UrlGeneratorInterface $urls,
        private readonly RequestStack $requests,
        private readonly Security $security
    ) {}

    #[LiveProp(writable: true)]
    public bool $include_reason = false;

    #[LiveProp]
    public ?int $user_id = null;

    public bool $is_edit { get => $this->user_id !== null; }


    /**
     * @return FormInterface
     * @throws UserNotFoundException
     * @throws InvalidOptionsException
     */
    protected function instantiateForm(): FormInterface
    {
        if(! $this->is_edit) {
            $dto = new UserDto();
        }
        else {
            $dto = $this->manager->getUserDto($this->user_id);
        }

        return $this->forms->createNamed(
            'form',                    // <-- force the name expected by the JS
            Form::class,
            $dto,
            [
                'include_reason' => $this->include_reason,
                'csrf_protection'  => true,
            ]
        );
    }

    /**
     * @return bool
     */
    protected function shouldSubmitFormOnRender(): bool
    {
        return false;
    }

    /**
     * @return RedirectResponse|null
     * @throws AccessDeniedException
     * @throws InvalidArgumentException
     * @throws InvalidParameterException
     * @throws MissingMandatoryParametersException
     * @throws RouteNotFoundException
     * @throws RuntimeException
     * @throws LogicException
     * @throws UnprocessableEntityHttpException
     */
    #[LiveAction]
    public function save(): ?RedirectResponse
    {
        if ($this->is_edit) {
            if (!$this->security->isGranted('permission', [
                'permission' => 'usr.edit',
                'user' => (int) $this->user_id,
                'user_edit' => 'details', // anything not 'roles'
            ])) {
                throw new AccessDeniedException();
            }
        } else {
            if (!$this->security->isGranted('permission', [
                'permission' => 'usr.add',
            ])) {
                throw new AccessDeniedException();
            }
        }

        $this->submitForm();

        if (!$this->form->isValid()) {
            return null; // Live will re-render with field errors
        }

        /** @var UserDto $dto */
        $dto   = $this->form->getData();

        $user = $this->getSecurityUser();

        $externalUser = null;

        if ($this->is_edit) {
            try {
                $externalUser = $this->manager->editExternalUser($dto, $this->user_id, $user->email);
                $this->persistFlash('success', 'User edited successfully');
            } catch (NoUpdateRequiredException|InternalUserException|NotBlankException $e) {
                $this->flash('warning', $e->getMessage() ?: 'Could not edit user');
            }
        }
        else {
            try {
                $externalUser = $this->manager->createExternalUser($dto, $user->email);
                $this->persistFlash('success', 'User added successfully');
            } catch (NotBlankException $e) {
                $this->flash('warning', $e->getMessage() ?: 'Could not add user');
            }
        }

        if($externalUser !== null) {
            return new RedirectResponse(
                $this->urls->generate('shared_user_view', ['id' => $externalUser->id])
            );
        }

        return null; //must be error
    }
}
