<?php
declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Twig\Components\User;

use Atlas\SecurityManagerBundle\Entity\User\User;
use Atlas\SecurityManagerBundle\Exception\LockException;
use Atlas\SecurityManagerBundle\Exception\User\InternalUserException;
use Atlas\SecurityManagerBundle\Exception\User\UserNotFoundException;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\SecurityManagerBundle\Service\User\UserManager;
use Atlas\SecurityManagerBundle\Twig\Components\Concern\FlashHelperTrait;
use Atlas\SecurityManagerBundle\Twig\Components\Concern\SecurityUserHelperTrait;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\ExpressionLanguage\Expression;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Security\Core\Exception\AccessDeniedException;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveArg;
use Symfony\UX\LiveComponent\Attribute\LiveProp;
use Symfony\UX\LiveComponent\ComponentToolsTrait;
use Symfony\UX\LiveComponent\DefaultActionTrait;
use Symfony\UX\LiveComponent\Attribute\LiveAction;

#[AsLiveComponent(
    name: 'user_details',
    template: '@SecurityManager/components/user/details.html.twig'
)]
final class UserDetailsComponent
{
    use DefaultActionTrait;
    use ComponentToolsTrait;
    use FlashHelperTrait;
    use SecurityUserHelperTrait;

    /** @var User */
    #[LiveProp]
    public User $user;

    #[LiveProp]
    public array $query_string = [];

    #[LiveProp]
    public string $variant = 'view';

    public bool $is_locked { get => $this->user->is_locked; }

    /**
     * @param UserManager $manager
     * @param Security $security
     * @param RequestStack $requests
     */
    public function __construct(
        private readonly UserManager $manager,
        private readonly Security $security,
        private readonly RequestStack $requests,
    ) {}

    /**
     * @param bool $lock
     * @param string $reason Reason supplied from modal
     * @throws AccessDeniedException
     */
    #[LiveAction]
    public function applyLock(
        #[LiveArg] bool $lock,
        #[LiveArg] string $reason
    ): void
    {
        if (!$this->security->isGranted('permission', [
            'permission' => 'usr.lock',
            'user' => $this->user->id,
        ])) {
            throw new AccessDeniedException();
        }

        $user = $this->getSecurityUser();

        try {
            $this->manager->lock($this->user, $user->email, $reason, $lock);
            $this->flash(
                'success',
                $lock ? 'User locked.' : 'User unlocked.'
            );

        } catch (LockException|UserNotFoundException|NotBlankException $e) {
            $this->flash('info', $e->getMessage() ?: 'No changes were made');
        }
    }

    /**
     * @param string $reason
     * @return void
     * @throws AccessDeniedException
     */
    #[LiveAction]
    #[IsGranted(
        attribute: 'permission',
        subject: [
            'permission' => new Expression('"usr.resend"')
        ]
    )]
    public function resendValidation(): void
    {

        $user = $this->getSecurityUser();

        try {

            $this->manager->resendValidation($this->user, $user->email);


            $this->flash(
                'success',
                'Validation email sent to user: ' . $this->user->email
            );
        } catch (UserNotFoundException $e) {
            $this->flash(
                'warning',
                'Cannot find user: ' . $e->getMessage()
            );
        } catch (InternalUserException $e) {
            $this->flash(
                'danger',
                'Internal users do not require validation: ' . $e->getMessage()
            );
        }
    }
}
