<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Controller;

use Atlas\SecurityManagerBundle\Exception\Security\InvalidClaimException;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\SecurityManagerBundle\Form\Security\PasswordRequestForm;
use Atlas\SecurityManagerBundle\Service\Security\PasswordManager;
use DateMalformedStringException;
use InvalidArgumentException;
use LogicException;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Form\Exception\OutOfBoundsException;
use Symfony\Component\Form\Exception\RuntimeException;
use Symfony\Component\Form\FormError;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Csrf\CsrfTokenManagerInterface;
use Symfony\Component\Security\Http\Authentication\AuthenticationUtils;

class SecurityController extends AbstractController
{

    public function __construct(
        #[Autowire('%security.loggedin.route%')] private readonly string $logged_in_route,
        private readonly PasswordManager $manager
    )
    {

    }

    /**
     * @throws ContainerExceptionInterface
     * @throws InvalidArgumentException
     * @throws NotFoundExceptionInterface
     */
    #[Route('/_debug/csrf', name: '_debug_csrf')]
    public function csrf(): Response
    {
        $tm = $this->container->get('security.csrf.token_manager');
        $token = $tm->getToken('authenticate')->getValue();
        return new Response('<pre>token: '.htmlspecialchars($token, ENT_QUOTES).'</pre>');
    }

    /**
     * @throws LogicException
     */
    #[Route(path: '/login', name: 'security_login')]
    public function login(AuthenticationUtils $authenticationUtils, CsrfTokenManagerInterface $csrfTokenManager): Response
    {

        if($this->isGranted('IS_AUTHENTICATED_FULLY')) {
            return $this->redirectToRoute($this->logged_in_route);
        }

        // get the login error if there is one
        $error = $authenticationUtils->getLastAuthenticationError();


        // last username entered by the user
        $lastUsername = $authenticationUtils->getLastUsername();

        $csrfTokenValue = $csrfTokenManager->getToken('authenticate')->getValue();


        return $this->render('@SecurityManager/security/login.html.twig', [
            'last_username' => $lastUsername,
            'error' => $error,
            '_csrf_token' => $csrfTokenValue,
        ]);
    }

    /**
     * @param Request $request
     * @return Response
     * @throws LogicException
     * @throws OutOfBoundsException
     * @throws RuntimeException
     */
    #[Route('/password/request', name: 'security_password_request', methods: ['GET','POST'])]
    public function requestPasswordReset(Request $request): Response
    {
        $form = $this->createForm(PasswordRequestForm::class);

        $form->handleRequest($request);

        if ($form->isSubmitted()) {
            if ($form->isValid()) {
                $username = (string) $form->get('username')->getData();

                try {
                    // Manager already avoids leaking account existence.
                    $this->manager->request($username);
                    $this->addFlash('success', 'If the account exists, we’ve emailed a reset link.');
                    return $this->redirectToRoute('security_login');
                } catch (InvalidClaimException) {
                    // Transient issue generating token; show generic message (no leak)
                    $this->addFlash('warning', 'We could not create a reset link right now. Please try again shortly.');
                } catch (NotBlankException $e) {
                    // Defensive: validator should already handle this, but keep as fallback.
                    $form->get('username')->addError(new FormError($e->getMessage() ?: 'Username or email cannot be blank'));
                }
            }
        }

        return $this->render('@SecurityManager/security/password_request.html.twig', [
            'form' => $form->createView(),
        ]);
    }

    /**
     * @param string $token
     * @return Response
     * @throws DateMalformedStringException
     * @throws LogicException
     */
    #[Route('/password/{token}', name: 'security_password_reset', methods: ['GET'])]
    public function allowPasswordReset(string $token): Response
    {
        if ($this->isGranted('IS_AUTHENTICATED_FULLY')) {
            return $this->redirectToRoute($this->logged_in_route);
        }

        $result = $this->manager->checkPasswordToken($token);

        if (is_array($result)) {
            // Invalid/expired/used token → show errors; no user, no form
            return $this->render('@SecurityManager/security/password_reset.html.twig', [
                'token' => $token,
                'errors' => $result,
                'show_form' => false,
                'user' => null,
            ]);
        }

        $user = $result;

        // Valid token → show form and pass the user object to the UX
        return $this->render('@SecurityManager/security/password_reset.html.twig', [
            'token' => $token,
            'errors' => [],
            'show_form' => true,
            'user' => $user,
        ]);
    }
}
