<?php

namespace Atlas\SecurityManagerBundle\Dto\User;

use Atlas\SecurityManagerBundle\Entity\User\User;
use Atlas\SecurityManagerBundle\Validator\InternalEmail;
use Atlas\SecurityManagerBundle\Validator\UniqueEntityField;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Validator\Context\ExecutionContextInterface;

class UserDto
{
    public ?int $id = null;

    #[Assert\NotBlank]
    #[Assert\Length(min: 2, max: 35)]
    public string $firstname = '';

    #[Assert\NotBlank]
    #[Assert\Length(min: 2, max: 35)]
    public string $lastname = '';

    #[Assert\NotBlank]
    #[Assert\Email]
    #[Assert\Length(max: 255)]
    #[InternalEmail]
    #[UniqueEntityField(entity: User::class, field: 'email', message: 'Email already exists.')]
    public string $email = '';

    public ?string $reason = '';

    #[Assert\Callback]
    public function validate(ExecutionContextInterface $context): void
    {
        $includeReason = $context->getRoot()->getConfig()->getOption('include_reason') ?? false;

        if ($includeReason) {
            if (empty($this->reason)) {
                $context->buildViolation('Reason is required.')
                    ->atPath('reason')
                    ->addViolation();
            } elseif (mb_strlen($this->reason, encoding: 'UTF-8') < 10 || mb_strlen($this->reason, encoding: 'UTF-8') > 255) {
                $context->buildViolation('Reason must be between 10 and 255 characters.')
                    ->atPath('reason')
                    ->addViolation();
            }
        }
    }
}
