<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Entity\Audit;

use Atlas\SecurityManagerBundle\Entity\User\User;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\SecurityManagerBundle\Repository\Audit\PasswordRepository;
use Atlas\AuditBundle\Attribute\NotLogged;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints\Length;
use Symfony\Component\Validator\Constraints\NotBlank;


#[NotLogged]
#[ORM\Entity(repositoryClass: PasswordRepository::class)]
#[ORM\Table(name: 'audit_password')]
#[ORM\Index(columns: ['user_id'])]
class Password
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\Column]
    private(set) int $user_id;

    #[ORM\Column(length: 60)]
    #[NotBlank]
    #[Length(max: 60)]
    private(set) string $password;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    private(set) DateTimeInterface $changed;

    #[ORM\Column(length: 255)]
    #[NotBlank]
    private(set) string $changed_by;

    #[ORM\JoinColumn(name: 'user_id', referencedColumnName: 'id')]
    #[ORM\ManyToOne(targetEntity: User::class)]
    private(set) User $user;

    /**
     * @param User $user
     * @param string $newHashedPassword
     * @param string $actionBy
     * @throws NotBlankException
     */
    public function __construct(User $user, string $newHashedPassword, string $actionBy)
    {
        if (empty (mb_trim($newHashedPassword, encoding: 'UTF-8'))) {
            throw new NotBlankException('Hashed password cannot be empty');
        }

        if (empty (mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw new NotBlankException('Action by cannot be empty');
        }

        $this->user = $user;
        $this->password = $newHashedPassword;
        $this->changed = new DateTimeImmutable();
        $this->changed_by = $actionBy;
    }

}
