<?php

namespace Atlas\SecurityManagerBundle\Entity\Role;

use Atlas\SecurityManagerBundle\Doctrine\Mapping\UppercaseCodeNormaliserTrait;
use Atlas\SecurityManagerBundle\Exception\ActivateException;
use Atlas\SecurityManagerBundle\Exception\NoUpdateRequiredException;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\AuditBundle\Attribute\AuditActor;
use Atlas\AuditBundle\Attribute\AuditTimestamp;
use Atlas\AuditBundle\Attribute\Enum\AuditActionType;
use Atlas\AuditBundle\Attribute\NotLogged;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Validator\Constraints\Length;
use Symfony\Component\Validator\Constraints\NotBlank;


use Atlas\SecurityManagerBundle\Repository\Role\RoleRepository;
#[ORM\Entity(repositoryClass: RoleRepository::class)]
#[ORM\Table(name: 'sys_role')]
#[UniqueEntity(fields: ['code'])]
#[ORM\HasLifecycleCallbacks]
class Role
{
    use UppercaseCodeNormaliserTrait;

    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\Column(length: 25, unique: true)]
    #[NotBlank]
    #[Length(min: 1, max: 25)]
    private(set) string $code;

    #[ORM\Column(length: 50)]
    #[NotBlank]
    #[Length(min: 2, max: 50)]
    private(set) string $name;

    #[ORM\Column]
    private(set) bool $active = false;

    #[NotLogged]
    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[AuditTimestamp(AuditActionType::INSERT)]
    private(set) DateTimeInterface $created;

    #[NotLogged]
    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[AuditActor(AuditActionType::INSERT)]
    private(set) string $created_by;

    #[NotLogged]
    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[AuditTimestamp(AuditActionType::UPDATE)]
    private(set) DateTimeInterface $modified;

    #[NotLogged]
    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[AuditActor(AuditActionType::UPDATE)]
    private(set) string $modified_by;

    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[Length(min: 10, max: 255)]
    private(set) string $reason;

    /** @var Collection<int, RolePermission> */
    #[ORM\OneToMany(targetEntity: RolePermission::class, mappedBy: 'role', orphanRemoval: true)]
    private(set) Collection $permissions;

    /** @var Collection<int, RoleForm> */
    #[ORM\OneToMany(targetEntity: RoleForm::class, mappedBy: 'role', orphanRemoval: true)]
    private(set) Collection $forms;

    /**
     * @param string $code
     * @param string $name
     * @param string $actionBy
     */
    public function __construct(string $code, string $name, string $actionBy)
    {
        if (empty($code = mb_trim($code, encoding: 'UTF-8'))) {
            throw new NotBlankException('Code should not be empty');
        }

        if (empty($name = mb_trim($name, encoding: 'UTF-8'))) {
            throw new NotBlankException('Name should not be empty');
        }

        if (empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw new NotBlankException('Actioned by should not be empty');
        }

        $this->code = mb_strtoupper($code, 'UTF-8');
        $this->name = $name;
        $this->reason = 'Initial creation of role';
        $this->created = $this->modified = new DateTimeImmutable();
        $this->created_by = $this->modified_by = $actionBy;

        $this->permissions = new ArrayCollection();
        $this->forms = new ArrayCollection();
    }

    /**
     * @param string $actionBy
     * @param string $reason
     * @param string|null $name
     * @return void
     */
    public function update(
        string $actionBy,
        string $reason,
        ?string $name = null
    ): void {
        $timestamp = new DateTimeImmutable();

        if (empty($reason = mb_trim($reason, encoding: 'UTF-8'))) {
            throw new NotBlankException('Reason should not be empty');
        }

        if (empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw new NotBlankException('Action by should not be empty');
        }

        $changed = false;

        if (!empty($name) && $this->name !== $name = mb_trim($name, encoding: 'UTF-8')) {
            $this->name = $name;
            $changed = true;
        }

        if ($changed) {
            $this->modified = $timestamp;
            $this->modified_by = $actionBy;
            $this->reason = $reason;
        }

        if(! $changed) throw NoUpdateRequiredException::forId('Role', $this->id);
    }

    /**
     * @param string $actionBy
     * @param string $reason
     * @param bool $activate
     * @return void
     */
    public function activate(string $actionBy, string $reason, bool $activate = true): void
    {
        if (empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw new NotBlankException('Actioned by should not be empty');
        }

        if (empty($reason = mb_trim($reason, encoding: 'UTF-8'))) {
            throw new NotBlankException('Reason should not be empty');
        }

        $timestamp = new DateTimeImmutable();

        if ($this->active !== $activate) {
            $this->active = $activate;
            $this->modified = $timestamp;
            $this->modified_by = $actionBy;
            $this->reason = $reason;
        }
        else {
            throw ActivateException::forResource('Role', $activate);
        }
    }

    //not database variables
    //virtual methods
    public bool $is_active {
        get => $this->active;
    }
}
