<?php

namespace Atlas\SecurityManagerBundle\Entity\Role;

use Atlas\AuditBundle\Attribute\AuditActor;
use Atlas\AuditBundle\Attribute\AuditTimestamp;
use Atlas\AuditBundle\Attribute\Enum\AuditActionType;
use Atlas\SecurityManagerBundle\Contract\FormInterface;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Validator\Constraints\Length;
use Symfony\Component\Validator\Constraints\NotBlank;


use Atlas\SecurityManagerBundle\Repository\Role\RoleFormRepository;
#[ORM\Entity(repositoryClass: RoleFormRepository::class)]
#[ORM\Table(name: 'sys_role_form')]
#[ORM\Index(columns: ['form_id'])]
#[ORM\Index(columns: ['role_id'])]
#[ORM\UniqueConstraint(columns: ['form_id', 'role_id'])]
#[UniqueEntity(fields: ['form', 'role'])]
class RoleForm
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\ManyToOne(targetEntity: Role::class, inversedBy: 'forms')]
    #[ORM\JoinColumn(name: 'role_id', referencedColumnName: 'id', nullable: false)]
    private(set) Role $role;

    #[ORM\ManyToOne(targetEntity: FormInterface::class)]
    #[ORM\JoinColumn(name: 'form_id', referencedColumnName: 'id', nullable: false)]
    private(set) FormInterface $form;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[AuditTimestamp(AuditActionType::INSERT)]
    #[AuditTimestamp(AuditActionType::UPDATE)]
    #[NotBlank]
    private(set) DateTimeInterface $granted;

    #[ORM\Column(length: 255)]
    #[AuditActor(AuditActionType::INSERT)]
    #[AuditActor(AuditActionType::UPDATE)]
    #[NotBlank]
    private(set) string $granted_by;

    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[Length(min: 10, max: 255)]
    private(set) string $reason;

    //not in db
    #[AuditTimestamp(AuditActionType::DELETE)]
    private(set) ?DateTimeInterface $deleted = null;
    #[AuditActor(AuditActionType::DELETE)]
    private(set) ?string $deleted_by = null;

    /**
     * @param Role $role
     * @param FormInterface $form
     * @param string $grantedBy
     * @param string $reason
     */
    public function __construct(Role $role, FormInterface $form, string $grantedBy, string $reason)
    {
        if (empty($grantedBy = mb_trim($grantedBy, encoding: 'UTF-8'))) {
            throw new NotBlankException('Granted by should not be empty');
        }

        if (empty($reason = mb_trim($reason, encoding: 'UTF-8'))) {
            throw new NotBlankException('Reason should not be empty');
        }

        $this->role = $role;
        $this->form = $form;
        $this->granted_by = $grantedBy;
        $this->reason = $reason;
        $this->granted = new DateTimeImmutable();
    }

    /**
     * @param string $deletedBy
     * @param string $reason
     * @return void
     */
    public function preDelete(string $deletedBy, string $reason): void
    {
        if (empty($deletedBy = mb_trim($deletedBy, encoding: 'UTF-8'))) {
            throw new NotBlankException('Deleted by should not be empty');
        }

        if (empty($reason = mb_trim($reason, encoding: 'UTF-8'))) {
            throw new NotBlankException('Reason should not be empty');
        }

        $this->deleted_by = $deletedBy;
        $this->reason = $reason;
        $this->deleted = new DateTimeImmutable();
    }
}
