<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Repository\Audit;

use Atlas\SecurityManagerBundle\Entity\Audit\Password;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @extends ServiceEntityRepository<Password>
 */
class PasswordRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Password::class);
    }

    /**
     * Get the most recent Password entity for a given user.
     *
     * @param int $userId the user's id
     * @return Password|null
     */
    public function getLatestPassword(int $userId): ?Password
    {
        return $this->findOneBy(
            ['user' => $userId],
            ['id'   => 'DESC']
        );
    }

    /**
     * Get the most recent password hash for a given user.
     *
     * @param int $userId the user's id
     * @return string|null
     * @throws NonUniqueResultException
     */
    public function getLatestPasswordHash(int $userId): ?string
    {
        $query = $this->createQueryBuilder('pwd')
            ->select('pwd.password')
            ->where('pwd.user = :userId')
            ->setParameter('userId', $userId)
            ->orderBy('pwd.id', 'DESC')
            ->setMaxResults(1);

        return $query
            ->getQuery()
            ->getOneOrNullResult(AbstractQuery::HYDRATE_SINGLE_SCALAR);
    }
    
    /**
     * @param int $userId
     * @param int $limit
     * @return array<int,string> Latest N hashes (DESC by id).
     */
    public function getRecentPasswordHashes(int $userId, int $limit = 6): array
    {
        $rows = $this->createQueryBuilder('pwd')
            ->select('pwd.password')
            ->where('pwd.user = :userId')
            ->setParameter('userId', $userId)
            ->orderBy('pwd.id', 'DESC')
            ->setMaxResults($limit)
            ->getQuery()
            ->getScalarResult();

        return array_map(static fn(array $r) => (string)$r['password'], $rows);
    }
}
