<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Security\Voter;

use Atlas\SecurityManagerBundle\Repository\Role\RoleFormRepository;
use Atlas\SecurityManagerBundle\Repository\Role\RolePermissionRepository;
use Atlas\SecurityManagerBundle\Security\User as SecurityUser;
use Psr\Cache\InvalidArgumentException;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Authorization\Voter\Vote;
use Symfony\Component\Security\Core\Authorization\Voter\Voter;

class PermissionVoter extends Voter
{
    const string PERMISSION = 'permission';

    public function __construct(
        private readonly RolePermissionRepository $permissions,
        private readonly RoleFormRepository $forms
    ) {}

    /**
     * @inheritDoc
     */
    protected function supports(string $attribute, mixed $subject): bool
    {
        //it must be of type permission
        if($attribute !== self::PERMISSION) return false;

        //subject must be an array
        if(! is_array($subject)) return false;

        //minimum must contain permission, but can contain location and form too
        if(! isset($subject['permission'])) return false;

        return true;
    }

    /**
     * @inheritDoc
     * @throws InvalidArgumentException
     */
    protected function voteOnAttribute(string $attribute, mixed $subject, TokenInterface $token, ?Vote $vote = null): bool
    {
        //get the user and then get their permissions
        $user = $token->getUser();

        // must be the correct user type (only one expected)
        if(! $user instanceof SecurityUser) return false;

        $permissions = $user->permissions;

        $appPermissions = $this->permissions->getPermissionMap();
        
        $appFormPermissions = $this->forms->getPermissionMap();

        $permission = (string) $subject['permission'];

        // Self rules only apply if caller provides the target user
        if (isset($subject['user'])) {

            $targetUserId = (int) $subject['user'];
            $currentUserId = (int) $user->id;

            if ($targetUserId === $currentUserId) {

                $requiredPermission = null;

                if ($permission === 'usr.lock' || $permission === 'usr.edit') {
                    $requiredPermission = 'usr.self';
                }

                if ($permission === 'usr.role') {
                    $requiredPermission = 'usr.role.self';
                }

                if ($requiredPermission !== null) {

                    $allowed = false;

                    foreach ($permissions as $roleCodes) {
                        if ($this->checkRole($roleCodes, $requiredPermission, $appPermissions)) {
                            $allowed = true;
                            break;
                        }
                    }

                    if (!$allowed) {
                        return false;
                    }
                }
            }
        }


        //must do role edit check first
        $isRoleCheck = $this->isRoleManagePermission($subject['permission']);

        if ($isRoleCheck) {
            if (!isset($subject['role'])) {
                return false;
            }

            $roleCode = (string) $subject['role'];

            // Is the user trying to manage a role they personally have?
            $userHasRole = false;

            foreach ($permissions as $roleCodes) {
                foreach ($roleCodes as $userRoleCode) {
                    if ($userRoleCode === $roleCode) {
                        $userHasRole = true;
                        break 2;
                    }
                }
            }

            // Self-management is only allowed if THAT role includes role.self
            if ($userHasRole) {
                if (!isset($appPermissions[$roleCode])) {
                    return false;
                }

                if (!in_array('role.self', $appPermissions[$roleCode], true)) {
                    return false;
                }
            }

            // do not return true here; normal permission logic below still applies
        }

        if (!isset($subject['location']) && !isset($subject['form'])) {
            if (array_any(
                $permissions,
                fn($roles) => $this->checkRole(
                    $roles,
                    $subject['permission'],
                    $appPermissions,
                    appFormPermissions: $appFormPermissions
                )
            )) {
                return true;
            }
        }

        if (!isset($subject['location']) && isset($subject['form'])) {
            $globalRoles = $permissions['all'] ?? [];
            if ($this->checkRole(
                $globalRoles,
                $subject['permission'],
                $appPermissions,
                form: $subject['form'],
                appFormPermissions: $appFormPermissions
            )) {
                return true;
            }
        }

        //otherwise it is more complex
        //if location is set
        if(isset($subject['location'])) {
            //we just look the locations and all

            if(isset($permissions['all'])) {

                if($this->checkRole(
                    $permissions['all'],
                    $subject['permission'],
                    $appPermissions,
                    form: $subject['form'] ?? null,
                    appFormPermissions: $appFormPermissions
                )
                ) {
                    return true;
                }
            }

            //then we check the exact location
            if(isset($permissions[$subject['location']])) {
                if($this->checkRole(
                    $permissions[$subject['location']],
                    $subject['permission'],
                    $appPermissions,
                    form: $subject['form'] ?? null,
                    appFormPermissions: $appFormPermissions
                )
                ) {
                    return true;
                }
            }
        }

        return false;
    }

    private function checkRole(
        array $roles,
        string $permission,
        array $appPermissions,
        ?string $form = null,
        array $appFormPermissions = []
    ): bool
    {

        foreach ($roles as $role) {
            if (!isset($appPermissions[$role])) {
                continue;
            }

            if (in_array($permission, $appPermissions[$role])) {
                if ($form !== null) {
                    $form = mb_strtoupper($form, encoding: 'UTF-8');
                    //we need to test form code too
                    if (isset($appFormPermissions[$role])) {
                        if (in_array($form, $appFormPermissions[$role])) {
                            return true;
                        }
                    }
                } else {
                    return true;
                }
            }
        }

        return false;
    }

    private function isRoleManagePermission(string $permission): bool
    {
        // be explicit: only block actions that actually "manage"
        return $permission === 'role.edit';
        // if you also want to block audit: add 'role.audit'
    }
}
