<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Service\User;

use Atlas\SecurityManagerBundle\Dto\User\LdapUserDto;
use Atlas\SecurityManagerBundle\Exception\User\InternalUserException;
use Atlas\SecurityManagerBundle\Exception\User\UserNotFoundException;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Ldap\Exception\ConnectionException;
use Symfony\Component\Ldap\Exception\LdapException;
use Symfony\Component\Ldap\Exception\NotBoundException;
use Symfony\Component\Ldap\LdapInterface;
use Symfony\Component\Validator\Constraints\Email;
use Symfony\Component\Validator\Exception\InvalidArgumentException;
use Symfony\Component\Validator\Exception\LogicException;
use Symfony\Component\Validator\Validator\ValidatorInterface;

class LdapSearch
{

    /* @var string[] */
    private array $internal_domains = [];

    /**
     * @param LdapInterface $ldap
     * @param ValidatorInterface $validator
     * @param bool $use_ldap
     * @param string|null $ldap_base_dn
     * @param string|null $ldap_search_dn
     * @param string|null $ldap_search_password
     * @param string|null $ldap_filter
     * @param array $ldap_attributes
     * @param array $internal_domains
     */
    public function __construct(
        private readonly LdapInterface $ldap,
        private readonly ValidatorInterface $validator,
        #[Autowire(param: 'security.ldap.directory.base_dn')] private readonly ?string $ldap_base_dn,
        #[Autowire(param: 'security.ldap.directory.search_dn')] private readonly ?string $ldap_search_dn,
        #[Autowire(param: 'security.ldap.directory.search_password')] private readonly ?string $ldap_search_password,
        #[Autowire(param: 'security.ldap.directory.filter')] private readonly ?string $ldap_filter,
        #[Autowire(param: 'security.ldap.enabled')] private readonly bool $use_ldap = false,
        #[Autowire(param: 'security.ldap.attributes')] private array $ldap_attributes = [],
        #[Autowire(param: 'security.user.internal_domains')] array $internal_domains = []
    )
    {
        if($this->use_ldap) {
            foreach (['username', 'givenname', 'surname', 'email'] as $k) {
                // Check existence first; avoid undefined index
                if (!array_key_exists($k, $this->ldap_attributes)) {
                    throw new LdapException(sprintf('security.ldap.attributes.%s must be configured', $k));
                }

                // Normalize
                $val = trim((string)$this->ldap_attributes[$k]);

                // Validate non-empty (multibyte-safe)
                if (mb_strlen($val) === 0) {
                    throw new LdapException(sprintf('security.ldap.attributes.%s must be configured', $k));
                }

                // Persist normalized value
                $this->ldap_attributes[$k] = $val;

            }
            if ($this->ldap_base_dn === null || trim($this->ldap_base_dn) === '') {
                throw new LdapException('security.ldap.directory.base_dn must be configured');
            }

            $this->internal_domains = array_values(array_unique(array_filter(array_map(
                static fn ($d) => mb_strtolower(trim((string) $d), 'UTF-8'),
                $internal_domains
            ), static fn ($d) => $d !== '')));

        }
    }

    /**
     * @param string $search
     * @return LdapUserDto
     * @throws ConnectionException
     * @throws InternalUserException
     * @throws InvalidArgumentException
     * @throws LdapException
     * @throws NotBlankException
     * @throws NotBoundException
     * @throws UserNotFoundException
     * @throws LogicException
     */
    public function search(string $search): LdapUserDto
    {
        if(! $this->use_ldap) throw new LdapException('LDAP must be enabled to add internal users');

        $attr_username = $this->ldap_attributes['username']  ?? 'uid';
        $attr_givenname = $this->ldap_attributes['givenname'] ?? 'givenName';
        $attr_surname = $this->ldap_attributes['surname']   ?? 'sn';
        $attr_email = $this->ldap_attributes['email']     ?? 'mail';

        $search = mb_trim($search, encoding: 'UTF-8');

        if($search === '') throw new NotBlankException('Search cannot be empty');

        $candidates = [];

        $push = static function (string $attr, string $val) use (&$candidates): void {
            $candidates[$attr."\0".$val] = [$attr, $val];
        };

        $constraints = [
            new Email([
                'mode' => Email::VALIDATION_MODE_STRICT, // RFC-compliant check
            ]),
        ];

        $isEmail = $this->validator->validate($search, $constraints);

        if($isEmail->count() === 0) {

            $atPosition = mb_strpos($search, '@', encoding: 'UTF-8');
            if($atPosition !== false) {
                $local = mb_substr($search, 0, $atPosition, encoding: 'UTF-8');
                $domain = mb_substr($search, $atPosition + 1, null, 'UTF-8');
                $domain = mb_strtolower(trim($domain), 'UTF-8');

                if(! in_array($domain, $this->internal_domains, strict: true)) throw new InternalUserException('Email must be an internal email address');

                $push($attr_email,$search);

                foreach($this->internal_domains as $domain) {
                    $push(
                        $attr_email,
                        sprintf('%s@%s', $local, $domain)
                    );
                }
            }
        }
        else {
            $push(
                $attr_username,
                $search
            );

            foreach($this->internal_domains as $domain) {
                $push(
                    $attr_username,
                    sprintf('%s@%s', $search, $domain)
                );
            }
        }

        if ($this->ldap_search_dn !== null && $this->ldap_search_dn !== '') {
            // Service account bind
            $this->ldap->bind($this->ldap_search_dn, (string) $this->ldap_search_password);
        } else {
            // Anonymous bind
            $this->ldap->bind();
        }

        $base_filter = trim((string) $this->ldap_filter);
        if ($base_filter !== '' && $base_filter[0] !== '(') {
            $base_filter = '(' . $base_filter . ')';
        }

        foreach($candidates as [$attr, $value]) {

            $escaped = $this->ldap->escape($value, flags: LdapInterface::ESCAPE_FILTER);

            $filter = sprintf('(%s=%s)', $attr, $escaped);

            if ($base_filter !== '') {
                $filter = sprintf('(&%s%s)', $base_filter, $filter);
            }

            $attrs = [$attr_username, $attr_givenname, $attr_surname, $attr_email];

            $cursor = $this->ldap->query($this->ldap_base_dn, $filter, [
                'maxItems'    => 2,
                'filter' =>  $attrs
            ])->execute();

            if($cursor->count() === 1) {

                $entry = $cursor->toArray()[0];

                $username = ($entry->getAttribute($attr_username)[0] ?? '') ?: '';
                $firstname = ($entry->getAttribute($attr_givenname)[0] ?? '') ?: '';
                $lastname = ($entry->getAttribute($attr_surname)[0] ?? '') ?: '';
                $email = ($entry->getAttribute($attr_email)[0] ?? '') ?: '';

                if ($username === '' || $email === '') {
                    // strict but clear failure mode
                    throw new InternalUserException('LDAP entry missing required attributes');
                }

                $user = new LdapUserDto();
                $user->username = $username;
                $user->firstname = $firstname;
                $user->lastname = $lastname;
                $user->email = $email;

                return $user;
            }
        }

        throw new UserNotFoundException('Could not find internal user');
    }
}