<?php
declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Service\User;

use Atlas\SecurityManagerBundle\Contract\EmailNotifierInterface;
use Atlas\SecurityManagerBundle\Contract\LocationInterface;
use Atlas\SecurityManagerBundle\Contract\LocationRepositoryInterface;
use Atlas\SecurityManagerBundle\Dto\User\LdapUserDto;
use Atlas\SecurityManagerBundle\Dto\User\UserDto;
use Atlas\SecurityManagerBundle\Dto\User\UserRoleDto;
use Atlas\SecurityManagerBundle\Entity\User\User;
use Atlas\SecurityManagerBundle\Entity\User\UserRole;
use Atlas\SecurityManagerBundle\Exception\LockException;
use Atlas\SecurityManagerBundle\Exception\NoUpdateRequiredException;
use Atlas\SecurityManagerBundle\Exception\Role\RoleException;
use Atlas\SecurityManagerBundle\Exception\Security\InvalidClaimException;
use Atlas\SecurityManagerBundle\Exception\User\InternalUserException;
use Atlas\SecurityManagerBundle\Exception\User\UserNotFoundException;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\SecurityManagerBundle\Repository\User\UserRepository;
use Atlas\SecurityManagerBundle\Repository\User\UserRoleRepository;
use Atlas\SecurityManagerBundle\Service\Security\JwtTokenGenerator;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\NonUniqueResultException;
use Modules\Paginator\Service\Paginator;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;

final readonly class UserManager
{
    public function __construct(
        #[Autowire('%security.home.route%')] private string $home_route,
        private EntityManagerInterface $entity_manager,
        private UserRepository $users,
        private UserRoleRepository $user_roles,
        private LocationRepositoryInterface $locations,
        private UsernameGenerator $username_generator,
        private Paginator $paginator,
        private JwtTokenGenerator $jwt_token_generator,
        private EmailNotifierInterface $mailer,
        private LoggerInterface $logger,
        private UrlGeneratorInterface $url_generator
    ) {
    }

    /**
     * @param array $options expected keys: c_f, c_o, c_d, c_s, p
     * @param int $perPage
     * @return array
     */
    public function getUsersPage(array $options, int $perPage = 50): array
    {
        $query = $this->users->findAllQueryBuilder(
            filter: $options['c_f'] ?? null,
            sort: $options['c_o'] ?? null,
            direction: $options['c_d'] ?? null,
            search: $options['c_s'] ?? null
        );

        $page = (int)($options['p'] ?? 1);
        if ($page < 1) {
            $page = 1;
        }

        $perPage = max(1, $perPage);

        return $this->paginator->paginate($query, $page, $perPage);
    }

    /**
     * Create Internal User from LDAP data
     * @throws NotBlankException
     */
    public function createInternalUser(
        LdapUserDto $user,
        string $actionBy
    ): User {
        $user = new User(
            $user->username,
            $user->firstname,
            $user->lastname,
            $user->email,
            $actionBy,
            internal: true
        );

        $user = $this->createUser($user);

        $link = $this->url_generator->generate(
            $this->home_route,
            [],
            UrlGeneratorInterface::ABSOLUTE_URL
        );

        $this->mailer->send(
            'INNEWUSER',
            $actionBy,
            [
                'username' => $user->username,
                'new_user_email' => $user->email,
                'firstname' => $user->firstname,
                'lastname' => $user->lastname,
                'link' => $link,
            ]
        );

        return $user;
    }

    private function createUser(User $user): User
    {
        $this->entity_manager->persist($user);
        $this->entity_manager->flush();

        return $user;
    }

    /**
     * @param UserDto $dto
     * @param string $actionBy
     * @return User
     * @throws NotBlankException
     * @throws InternalUserException
     */
    public function createExternalUser(UserDto $dto, string $actionBy): User
    {
        $user = new User(
            $this->username_generator->generate($dto->firstname, $dto->lastname),
            $dto->firstname,
            $dto->lastname,
            $dto->email,
            $actionBy
        );

        $user = $this->createUser($user);

        return $this->sendNewUserEmail($user, $actionBy);
    }

    /**
     * @param int|User $user
     * @param string $actionBy
     * @return User
     * @throws InternalUserException
     * @throws UserNotFoundException
     */
    public function resendValidation(int|User $user, string $actionBy): User
    {
        if(is_int($user)) {
            $user = $this->users->findOrThrow($user);
        }

        if($user->internal) throw new InternalUserException('User cannot be an internal user');

        return $this->sendNewUserEmail($user, $actionBy);
    }

    /**
     * @param UserDto $dto
     * @param int|User $user
     * @param string $actionBy
     * @return User
     * @throws InternalUserException
     * @throws NoUpdateRequiredException
     * @throws NotBlankException
     * @throws UserNotFoundException
     */
    public function editExternalUser(UserDto $dto, int|User $user, string $actionBy): User
    {
        if(is_int($user)) {
            $user = $this->users->findOrThrow($user);
        }

        $user->update(
            $actionBy,
            $dto->reason,
            firstname: $dto->firstname,
            lastname: $dto->lastname,
            email: $dto->email
        );

        $this->entity_manager->flush();

        return $user;
    }

    /**
     * Get accessible locations for a username.
     *
     * @param string $username
     * @param string|null $permission
     * @param bool $includeChildren
     * @param bool $asEntities
     * @param bool $activeOnly
     * /**
     * @return int[]|LocationInterface[]  IDs when $asEntities=false; Location[] when $asEntities=true
     */
    public function getUserLocations(
        string $username,
        ?string $permission = null,
        bool $includeChildren = true,
        bool $asEntities = false,
        bool $activeOnly = false
    ): array {
        $user = $this->users->findOneByUsername($username);
        if (!$user) {
            return [];
        }

        $direct = $this->user_roles->findDirectlyAssignedLocations(
            userId: $user->id,
            permissionCode: $permission,
            asEntities: $asEntities
        );

        // ALL locations
        if ($direct->is_all) {
            if ($asEntities) {
                return $this->locations
                    ->findAllQueryBuilder(
                        filter: $activeOnly ? 'active' : null,
                        sort: 'name',
                        direction: 'asc'
                    )
                    ->getQuery()
                    ->getResult(); // Location[]
            }

            return $this->locations->findAllLocationIds($activeOnly); // int[]
        }

        // Direct (no ALL)
        $items = $direct->items;
        if (!$includeChildren) {
            return $items; // already IDs or Location[]
        }

        if ($asEntities) {
            /** @var LocationInterface[] $directLocations */
            $directLocations = $items;
            $ids = array_map(static fn(LocationInterface $s): int => $s->id, $directLocations);
            if (!$ids) {
                return [];
            }

            return $this->locations->findChildLocations($ids, returnEntities: true); // Location[]
        }

        /** @var int[] $ids */
        $ids = $items;
        if (!$ids) {
            return [];
        }

        return $this->locations->findChildLocations($ids); // int[]
    }

    /**
     * @param int|User $user
     * @param string $actionBy
     * @param string $reason
     * @param bool $lock
     * @return User
     * @throws LockException
     * @throws NotBlankException
     * @throws UserNotFoundException
     */
    public function lock(
        int|User $user,
        string $actionBy,
        string $reason,
        bool $lock = true
    ): User
    {
        if(is_int($user)) {
            $user = $this->users->findOrThrow($user);
        }

        if ($user->is_locked === $lock) {
            throw new LockException(
                $lock ? sprintf('User: %s is already locked', $user->email) :
                    sprintf('User: %s is already unlocked',$user->email)
            );
        }

        $user->lock($actionBy, $reason, lock: $lock);

        $this->entity_manager->flush();

        return $user;
    }

    /**
     * @param int $id
     * @return UserDto
     * @throws UserNotFoundException
     */
    public
    function getUserDto(
        int $id
    ): UserDto {
        $user = $this->users->findOrThrow($id);

        $dto = new UserDto();
        $dto->id = $user->id;
        $dto->firstname = $user->firstname;
        $dto->lastname = $user->lastname;
        $dto->email = $user->email;

        return $dto;
    }

    /**
     * @param int|User $user
     * @param UserRoleDto $dto
     * @param string $actionBy
     * @return void
     * @throws NonUniqueResultException
     * @throws NotBlankException
     * @throws RoleException
     * @throws UserNotFoundException
     */
    public function addUserRole(int|User $user, UserRoleDto $dto, string $actionBy): void
    {
        if(is_int($user)) {
            $user = $this->users->findOrThrow($user);
        }

        if ($dto->location !== null) {
            if ($this->user_roles->existsFor($user->id, $dto->role->id, locationId: $dto->location->id)) {
                throw new RoleException('This assignment already exists.');
            }
        }

        if ($this->user_roles->existsFor($user->id, $dto->role->id)) {
            throw new RoleException('A global assignment for this role already exists.');
        }

        $userRole = new UserRole($user, $dto->role, $actionBy, $dto->reason, location: $dto->location);

        if ($dto->location === null) {
            $noneGlobal = $this->user_roles->findForUserOrdered(
                $user->id,
                roleId: $dto->role->id,
                find: $this->user_roles::JUST_LOCATIONS
            );
            foreach ($noneGlobal as $noneGlobalRole) {
                $noneGlobalRole->preDelete('Automatically removed when ALL added to role', $actionBy);
                $this->entity_manager->remove($noneGlobalRole);
                $this->entity_manager->flush();
            }
        }

        $this->entity_manager->persist($userRole);
        $this->entity_manager->flush();
    }

    /**
     * @param UserRole $userRole
     * @param string $reason
     * @param string $actionBy
     * @return void
     * @throws NotBlankException
     */
    public function removeUserRole(UserRole $userRole, string $reason, string $actionBy): void
    {
        $userRole->preDelete($actionBy, $reason);

        $this->entity_manager->remove($userRole);
        $this->entity_manager->flush();
    }

    /**
     * @param User $user
     * @param string $actionBy
     * @return User
     */
    private function sendNewUserEmail(User $user, string $actionBy): User
    {
        try {
            $token = $this->jwt_token_generator->issuePasswordReset($user->username);
        } catch (NotBlankException|InvalidClaimException $e) {
            $this->logger->error('Failed to issue password reset token', [
                'user' => $user->username,
                'error' => $e->getMessage(),
            ]);
            throw new InternalUserException('Failed to issue password reset token.', 0, $e);
        }

        $link = $this->url_generator->generate(
            'security_password_reset',
            ['token' => $token],
            UrlGeneratorInterface::ABSOLUTE_URL
        );

        $this->mailer->send(
            'NEWUSER',
            $actionBy,
            [
                'username' => $user->username,
                'new_user_email' => $user->email,
                'firstname' => $user->firstname,
                'lastname' => $user->lastname,
                'link' => $link
            ]
        );

        return $user;
    }
}
