<?php
declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Twig\Components\Concern;

use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\UX\LiveComponent\ComponentToolsTrait;

trait FlashHelperTrait
{

    /**
     * Emit an ephemeral flash handled by Stimulus.
     *
     * @param string      $type            success|error|danger|warning|notice|info
     * @param string      $message
     * @param string      $placement       'global' (base) or 'local' (component)
     * @param string|null $bootstrapClass  optional explicit Bootstrap class
     */
    private function flash(
        string $type,
        string $message,
        string $placement = 'global',
        ?string $bootstrapClass = null,
    ): void {
        $payload = [
            'type'      => $type,
            'message'   => $message,
            'placement' => $placement,          // 'global' or 'local'
        ];

        if ($bootstrapClass !== null) {
            // snake_case payload key per your standard
            $payload['bootstrap_class'] = $bootstrapClass;
        }

        // LiveComponent provides emit(); controllers don't use this trait
        $this->dispatchBrowserEvent('flash:add', $payload);
    }

    /** Convenience for local placement (near the component). */
    private function flashLocal(string $type, string $message, ?string $bootstrapClass = null): void
    {
        $this->flash($type, $message, 'local', $bootstrapClass);
    }

    /** Convenience for global/base placement. */
    private function flashGlobal(string $type, string $message, ?string $bootstrapClass = null): void
    {
        $this->flash($type, $message, 'global', $bootstrapClass);
    }

    private function persistFlash(string $type, string $message): void
    {
        $req = $this->requests->getCurrentRequest();
        if ($req && $req->hasSession()) {
            $req->getSession()->getFlashBag()->add($type, $message);
        }
    }
}
